package ru.foodtechlab.lib.auth.integration.restapi.feign.authorization.impl;

import com.rcore.rest.api.commons.exception.HttpCommunicationException;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Value;
import ru.foodtechlab.lib.auth.integration.core.AccessTokenService;
import ru.foodtechlab.lib.auth.integration.core.FoodtechlabGroupProperties;
import ru.foodtechlab.lib.auth.integration.restapi.feign.authorization.factory.FeignAuthorizationClientFactory;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.requests.RefreshTokenRequest;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.requests.UsernameAuthorizationRequest;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.responses.TokenPairDTO;
import ru.foodtechlab.lib.auth.service.facade.authorizationSession.dto.responses.AuthorizationSessionResponse;

import java.util.HashMap;
import java.util.Map;

@RequiredArgsConstructor
public class AccessTokenServiceWithGroups implements AccessTokenService {

    private final FoodtechlabGroupProperties groups;
    private final FeignAuthorizationClientFactory httpClientFactory;
    private final Map<String, TokenPairDTO> tokens = new HashMap<>();

    @Value("${spring.application.name}")
    private String serviceName;
    private final String roleCode = "MICROSERVICE";
    private final String platform = "spring-boot-app";

    @Override
    public String getAccessToken(String requestUrl) {
        var group = findGroup(requestUrl);
        var targetTokenPair = tokens.get(group.getName());
        if (targetTokenPair == null)
            tokens.put(group.getName(), login(group));
        else if (targetTokenPair.getAccessToken() == null) {
            if (targetTokenPair.getRefreshToken() != null)
                tokens.put(group.getName(), refreshToken(group));
            else
                tokens.put(group.getName(), login(group));
        }

        return tokens.get(group.getName()).getAccessToken();
    }

    @Override
    public void clearToken(String requestUrl) {
        var group = findGroup(requestUrl);
        var targetTokenPair = tokens.get(group.getName());
        targetTokenPair.setAccessToken(null);
    }

    private TokenPairDTO login(FoodtechlabGroupProperties.Group group) {
        var httpClient = httpClientFactory.create(group.getAuthService().getUrl());
        var newPair = httpClient.usernameAuthorization(UsernameAuthorizationRequest.builder()
                .username(group.getUsername())
                .password(group.getPassword())
                .roleCode(roleCode)
                .ip("127.0.0.1")
                .deviceId(serviceName)
                .applicationDetails(AuthorizationSessionResponse.ApplicationDetails.builder()
                        .name(serviceName)
                        .platform(platform)
                        .build())
                .build())
                .getResult();
        return TokenPairDTO.of(newPair.getAccessToken(), newPair.getRefreshToken());
    }

    private TokenPairDTO refreshToken(FoodtechlabGroupProperties.Group group) {
        try {
            var targetTokenPair = tokens.get(group.getName());
            var httpClient = httpClientFactory.create(group.getAuthService().getUrl());
            var newPair = httpClient.refreshToken(RefreshTokenRequest.of(targetTokenPair.getRefreshToken())).getResult();
            return TokenPairDTO.of(newPair.getAccessToken(), newPair.getRefreshToken());
        } catch (HttpCommunicationException e) {
            return login(group);
        }
    }

    private FoodtechlabGroupProperties.Group findGroup(String requestUrl) {
        return groups.findGroupByService(requestUrl)
                .orElseThrow(() -> new RuntimeException("Service group not found. Please, configure your groups"));
    }
}
