package ru.foodtechlab.lib.auth.integration.proxy.api.authorization.v1.controllers;

import com.rcore.rest.api.commons.response.OkApiResponse;
import com.rcore.rest.api.commons.response.SuccessApiResponse;
import lombok.RequiredArgsConstructor;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import ru.foodtechlab.lib.auth.integration.core.authorizartion.AuthorizationServiceFacade;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.requests.*;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.responses.BasicAuthorizationResponse;
import ru.foodtechlab.lib.auth.service.facade.authorization.dto.responses.ConfirmationCodeTimerResponse;
import ru.foodtechlab.lib.auth.service.facade.authorization.resources.AuthorizationResource;
import ru.foodtechlab.lib.auth.service.facade.authorization.routers.AuthorizationRoutes;
import ru.foodtechlab.lib.auth.service.facade.authorizationSession.dto.responses.AuthorizationSessionResponse;

import javax.servlet.http.HttpServletRequest;

@RequiredArgsConstructor
public class AuthorizationProxyApiController implements AuthorizationResource {
    private final AuthorizationServiceFacade authorizationServiceFacade;


    @Override
    @GetMapping({AuthorizationRoutes.LOGOUT})
    public OkApiResponse logout(@RequestParam String accessToken) {
        LogoutRequest request = new LogoutRequest(accessToken);
        authorizationServiceFacade.logout(request);
        return OkApiResponse.of();
    }

    @Override
    @PostMapping(value = AuthorizationRoutes.REFRESH)
    public SuccessApiResponse<BasicAuthorizationResponse> refreshToken(@RequestBody RefreshTokenRequest refreshTokenRequest) {
        return SuccessApiResponse.of(authorizationServiceFacade.refreshToken(refreshTokenRequest));
    }

    @Override
    @Deprecated(forRemoval = true)
    @PostMapping(AuthorizationRoutes.LOGIN_PASSWORD_EMAIL)
    public SuccessApiResponse<BasicAuthorizationResponse> emailAuthorization(@RequestBody BasicEmailAuthorizationRequest request) {
        HttpServletRequest httpServletRequest = ((ServletRequestAttributes) RequestContextHolder
                .getRequestAttributes())
                .getRequest();

        EmailAuthorizationRequest emailAuthorizationRequest =
                EmailAuthorizationRequest
                        .of(request.getEmail(), request.getPassword());

        emailAuthorizationRequest.setIp(httpServletRequest.getRemoteAddr());
        emailAuthorizationRequest.setDeviceId("proxy-support-legacy-device-id");
        emailAuthorizationRequest.setApplicationDetails(
                AuthorizationSessionResponse.ApplicationDetails.builder()
                        .name("proxy-support-legacy-device")
                        .platform("consumer")
                        .versionName("legacy")
                        .build()
        );
        emailAuthorizationRequest.setIsRegistrationAllowed(false);

        return SuccessApiResponse.of(authorizationServiceFacade.emailAuthorization(emailAuthorizationRequest));
    }

    @Override
    @PostMapping({AuthorizationRoutes.LOGIN_PASSWORD_USERNAME})
    public SuccessApiResponse<BasicAuthorizationResponse> usernameAuthorization(@RequestBody BasicUsernameAuthorizationRequest request) {
        HttpServletRequest httpServletRequest = ((ServletRequestAttributes) RequestContextHolder
                .getRequestAttributes())
                .getRequest();

        UsernameAuthorizationRequest usernameAuthorizationRequest =
                UsernameAuthorizationRequest
                        .of(request.getUsername(), request.getPassword());

        usernameAuthorizationRequest.setIp(httpServletRequest.getRemoteAddr());
        usernameAuthorizationRequest.setDeviceId("proxy-support-legacy-device-id");
        usernameAuthorizationRequest.setApplicationDetails(
                AuthorizationSessionResponse.ApplicationDetails.builder()
                        .name("proxy-support-legacy-device")
                        .platform("consumer")
                        .versionName("legacy")
                        .build()
        );
        usernameAuthorizationRequest.setIsRegistrationAllowed(false);

        return SuccessApiResponse.of(authorizationServiceFacade.usernameAuthorization(usernameAuthorizationRequest));
    }


    @Override
    @PostMapping({AuthorizationRoutes.SINGLE_FACTOR_ONE_STEP_WITH_PASSWORD_EMAIL})
    public SuccessApiResponse<BasicAuthorizationResponse> emailAuthorization(@RequestBody EmailAuthorizationRequest request) {
        return SuccessApiResponse.of(authorizationServiceFacade.emailAuthorization(request));
    }


    @Override
    @PostMapping({AuthorizationRoutes.SINGLE_FACTOR_ONE_STEP_WITH_PASSWORD_PHONE})
    public SuccessApiResponse<BasicAuthorizationResponse> phoneAuthorization(@RequestBody PhoneNumberAuthorizationRequest request) {
        return SuccessApiResponse.of(authorizationServiceFacade.phoneNumberAuthorization(request));
    }

    @Override
    @PostMapping({AuthorizationRoutes.SINGLE_FACTOR_ONE_STEP_WITH_PASSWORD_USERNAME})
    public SuccessApiResponse<BasicAuthorizationResponse> usernameAuthorization(@RequestBody UsernameAuthorizationRequest request) {
        return SuccessApiResponse.of(authorizationServiceFacade.usernameAuthorization(request));
    }

    @Override
    @PostMapping({AuthorizationRoutes.LOGOUT})
    public OkApiResponse logout(@RequestBody LogoutRequest request) {
        authorizationServiceFacade.logout(request);
        return OkApiResponse.of();
    }

    @Override
    @PostMapping(value = AuthorizationRoutes.SINGLE_FACTOR_TWO_STEPS_WITH_CONFIRMATION_EMAIL_CONFIRM)
    public SuccessApiResponse<BasicAuthorizationResponse> signUpConfirmationByEmail(@RequestBody SignUpConfirmationByEmailRequest request) {
        return SuccessApiResponse.of(authorizationServiceFacade.signUpConfirmationByEmail(request));
    }

    @Override
    @PostMapping(value = AuthorizationRoutes.SINGLE_FACTOR_TWO_STEPS_WITH_CONFIRMATION_EMAIL_INIT)
    public SuccessApiResponse<ConfirmationCodeTimerResponse> signUpByEmail(@RequestBody SignUpByEmailRequest request) {
        return SuccessApiResponse.of(authorizationServiceFacade.singUpByEmail(request));
    }

    @Override
    @PostMapping(value = AuthorizationRoutes.SINGLE_FACTOR_TWO_STEPS_WITH_CONFIRMATION_PHONE_CONFIRM)
    public SuccessApiResponse<BasicAuthorizationResponse> signUpConfirmationByPhoneNumber(@RequestBody SignUpConfirmationByPhoneNumberRequest request) {
        return SuccessApiResponse.of(authorizationServiceFacade.signUpConfirmationByPhoneNumber(request));
    }

    @Override
    @PostMapping(value = AuthorizationRoutes.SINGLE_FACTOR_TWO_STEPS_WITH_CONFIRMATION_PHONE_INIT)
    public SuccessApiResponse<ConfirmationCodeTimerResponse> signUpByPhoneNumber(@RequestBody SignUpByPhoneNumberRequest request) {
        return SuccessApiResponse.of(authorizationServiceFacade.singUpByPhoneNumber(request));
    }
}
