package ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.migration.port;

import com.rcore.database.mongo.commons.query.FindByIdQuery;
import com.rcore.database.mongo.commons.utils.CollectionNameUtils;
import com.rcore.domain.commons.port.dto.SearchResult;
import lombok.RequiredArgsConstructor;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;
import ru.foodtechlab.lib.auth.service.domain.migration.entity.MigrationEntity;
import ru.foodtechlab.lib.auth.service.domain.migration.port.MigrationRepository;
import ru.foodtechlab.lib.auth.service.domain.migration.port.filters.MigrationFilters;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.migration.documents.MigrationDoc;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.migration.manualMigrations.v3_1_7.migration.MigrateToV3_1_7;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.migration.mapper.MigrationMapper;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.migration.query.FindWithFiltersQuery;

import java.util.Optional;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Repository
public class MongoMigrationRepository implements MigrationRepository {
    private static final String collectionName = CollectionNameUtils.getCollectionName(MigrationDoc.class);

    private final MongoTemplate mongoTemplate;
    private final MigrationMapper mapper;

    @Override
    public MigrationEntity save(MigrationEntity entity) {
        return mapper.inverseMap(mongoTemplate.save(mapper.map(entity), collectionName));
    }

    @Override
    public Boolean delete(String id) {
        long deleteCount = mongoTemplate.remove(FindByIdQuery.of(id).getQuery(), collectionName).getDeletedCount();

        return deleteCount > 0;
    }

    @Override
    public Optional<MigrationEntity> findById(String s) {
        return Optional.ofNullable(mongoTemplate.findById(s, MigrationDoc.class)).map(mapper::inverseMap);
    }

    @Override
    public SearchResult<MigrationEntity> find(MigrationFilters filters) {
        Query query = new FindWithFiltersQuery(filters).getQuery();

        return SearchResult.withItemsAndCount(
                mongoTemplate.find(query, MigrationDoc.class).stream().map(mapper::inverseMap).collect(Collectors.toList()),
                mongoTemplate.count(query.limit(0).skip(0), MigrationDoc.class)
        );
    }

    @Override
    public Long count() {
        return mongoTemplate.count(new Query(), MigrationDoc.class);
    }

    @Override
    public MigrationEntity getMigration() {
        Optional<MigrationEntity> opt = findById(MigrationEntity.ID);
        if (opt.isPresent()) return opt.get();
        MigrationEntity migrationEntity = new MigrationEntity(MigrationEntity.ID);
        migrationEntity.setVersion(0);
        save(migrationEntity);
        return migrationEntity;
    }

    @Override
    public MigrationEntity runMigration() {
        MigrationEntity migrationEntity = getMigration();

        MigrateToV3_1_7 migrateToV3_1_7 = new MigrateToV3_1_7(mongoTemplate, this);

        if (migrateToV3_1_7.checkAvailable(migrationEntity) == true) {
            migrationEntity = migrateToV3_1_7.checkAndRunMigration(migrationEntity);
        }

        return migrationEntity;
    }

    @Override
    public boolean exist(String s) {
        return false;
    }
}
