package ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.credential.port;

import com.rcore.database.mongo.commons.query.AbstractExampleQuery;
import com.rcore.database.mongo.commons.utils.CollectionNameUtils;
import com.rcore.domain.commons.port.dto.SearchResult;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;
import ru.foodtechlab.abe.mongo.port.AbstractMongoSafeDeleteRepository;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.LoginDetails;
import ru.foodtechlab.lib.auth.service.domain.credential.entity.CredentialEntity;
import ru.foodtechlab.lib.auth.service.domain.credential.filters.SearchByRoleFilters;
import ru.foodtechlab.lib.auth.service.domain.credential.port.CredentialRepository;
import ru.foodtechlab.lib.auth.service.domain.credential.port.filters.CredentialFilters;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.credential.mapper.CredentialDocMapper;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.credential.model.CredentialDoc;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.credential.query.*;

import java.util.Optional;
import java.util.stream.Collectors;

@Repository
public class MongoCredentialRepository extends AbstractMongoSafeDeleteRepository<String, CredentialEntity, CredentialDoc, CredentialFilters> implements CredentialRepository {

    public MongoCredentialRepository(CredentialDocMapper mapper, MongoTemplate mongoTemplate) {
        super(CredentialDoc.class, mapper, mongoTemplate);
    }

    @Override
    public Optional<CredentialEntity> findByUsername(String username) {
        return Optional.ofNullable(mongoTemplate.findOne(FindByUsernameQuery.of(username).getQuery(), CredentialDoc.class))
                .map(mapper::inverseMap);
    }

    @Override
    public Optional<CredentialEntity> findByEmail(String email) {
        return Optional.ofNullable(mongoTemplate.findOne(FindByEmailQuery.of(email).getQuery(), CredentialDoc.class))
                .map(mapper::inverseMap);
    }

    @Override
    public SearchResult<CredentialEntity> findByRole(SearchByRoleFilters filters) {
        Query query = new FindWithFiltersRoleQuery(filters).getQuery();
        return SearchResult.withItemsAndCount(
                mongoTemplate.find(query, CredentialDoc.class).stream().map(mapper::inverseMap).collect(Collectors.toList()),
                mongoTemplate.count(query.limit(0).skip(0), CredentialDoc.class)
        );
    }

    @Override
    public Optional<CredentialEntity> findByPhone(String phone) {
        return Optional.ofNullable(mongoTemplate.findOne(FindByPhoneQuery.of(phone).getQuery(), CredentialDoc.class))
                .map(mapper::inverseMap);
    }

    @Override
    public Optional<CredentialEntity> findByLoginDetails(LoginDetails loginDetails) {
        return Optional.ofNullable(mongoTemplate.findOne(FindByLoginDetailsQuery.of(loginDetails).getQuery(), CredentialDoc.class))
                .map(mapper::inverseMap);
    }

    @Override
    protected AbstractExampleQuery getSearchQuery(CredentialFilters filters) {
        return new FindCredentialWithFiltersQuery(filters);
    }

    public static String getCollectionName() {
        return CollectionNameUtils.getCollectionName(CredentialDoc.class);
    }
}

