package ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.authorizationSession.mapper;

import com.rcore.commons.mapper.ExampleDataMapper;
import lombok.RequiredArgsConstructor;
import org.bson.types.ObjectId;
import org.springframework.stereotype.Component;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.AuthSessionEntity;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.ClientInfo;
import ru.foodtechlab.lib.auth.service.domain.auth.entity.LoginDetails;
import ru.foodtechlab.lib.auth.service.domain.auth.port.AuthorizationSessionIdGenerator;
import ru.foodtechlab.lib.auth.service.domain.credential.entity.PhoneNumber;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.authorizationSession.model.AuthSessionDoc;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.credential.mapper.CredentialDocMapper;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.role.mapper.RoleDocMapper;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.token.mapper.AccessTokenDocMapper;
import ru.foodtechlab.lib.auth.service.infrastructure.database.mongo.token.mapper.RefreshTokenDocMapper;

import java.util.Optional;

@RequiredArgsConstructor
@Component
public class AuthorizationSessionDocMapper implements ExampleDataMapper<AuthSessionEntity, AuthSessionDoc> {

    private final AuthorizationSessionIdGenerator<ObjectId> authorizationSessionIdGenerator;
    private final CredentialDocMapper credentialDocMapper;
    private final RoleDocMapper roleDocMapper;
    private final RefreshTokenDocMapper refreshTokenDocMapper;
    private final AccessTokenDocMapper accessTokenDocMapper;

    @Override
    public AuthSessionEntity inverseMap(AuthSessionDoc doc) {
        return AuthSessionEntity.builder()
                .id(doc.getId().toString())
                .createdAt(doc.getCreatedAt())
                .updatedAt(doc.getUpdatedAt())
                .isDeleted(doc.isDeleted())
                .clientInfo(ClientInfo.builder()
                        .deviceId(doc.getDeviceId())
                        .ip(doc.getIpV4())
                        .application(Optional.ofNullable(doc.getApplicationDetails())
                                .map(AuthSessionDoc.ApplicationDetails::getName)
                                .orElse(null))
                        .versionName(Optional.ofNullable(doc.getApplicationDetails())
                                .map(AuthSessionDoc.ApplicationDetails::getVersionName)
                                .orElse(null))
                        .platform(Optional.ofNullable(doc.getApplicationDetails())
                                .map(AuthSessionDoc.ApplicationDetails::getPlatform)
                                .orElse(null))
                        .build())
                .loginDetails(LoginDetails.builder()
                        .email(doc.getEmail())
                        .username(doc.getUsername())
                        .phoneNumber(Optional.ofNullable(doc.getPhoneNumber())
                                .map(p -> new PhoneNumber(doc.getPhoneNumber(), doc.getIsoTwoLetterCountryCode()))
                                .orElse(null))
                        .build())
                .loginType(doc.getLoginType())
                .errors(doc.getErrors())
                .ttl(doc.getTtl())
                .expiredAt(doc.getExpireAt())
                .confirmationCodeType(doc.getConfirmationCodeType())
                .isRegistrationAllowed(doc.getIsRegistrationAllowed())
                .authSessionLoginInitAttempts(doc.getAuthSessionLoginInitAttempts())
                .status(doc.getStatus())
                .type(doc.getType())
                .authSessionLoginConfirmAttemptsLimit(doc.getAuthSessionLoginConfirmAttemptsLimit())
                .authSessionLoginConfirmAttempts(doc.getAuthSessionLoginConfirmAttempts())
                .credential(Optional.ofNullable(doc.getCredential())
                        .map(credentialDocMapper::inverseMap)
                        .orElse(null))
                .targetRole(Optional.ofNullable(doc.getRole())
                        .map(roleDocMapper::inverseMap)
                        .orElse(null))
                .accessToken(Optional.ofNullable(doc.getAccessToken())
                        .map(accessTokenDocMapper::inverseMap)
                        .orElse(null))
                .refreshToken(Optional.ofNullable(doc.getRefreshToken())
                        .map(refreshTokenDocMapper::inverseMap)
                        .orElse(null))
                .version(doc.getVersion())
                .build();
    }

    @Override
    public AuthSessionDoc map(AuthSessionEntity entity) {
        return AuthSessionDoc.builder()
                .id(authorizationSessionIdGenerator.parse(entity.getId()))
                .createdAt(entity.getCreatedAt())
                .updatedAt(entity.getUpdatedAt())
                .deleted(entity.isDeleted())
                .loginType(entity.getLoginType())
                .errors(entity.getErrors())
                .ttl(entity.getTtl())
                .authSessionLoginInitAttempts(entity.getAuthSessionLoginInitAttempts())
                .authSessionLoginConfirmAttemptsLimit(entity.getAuthSessionLoginConfirmAttemptsLimit())
                .authSessionLoginConfirmAttempts(entity.getAuthSessionLoginConfirmAttempts())
                .expireAt(entity.getExpiredAt())
                .confirmationCodeType(entity.getConfirmationCodeType())
                .isRegistrationAllowed(entity.isRegistrationAllowed())
                .status(entity.getStatus())
                .type(entity.getType())
                .email(entity.getLoginDetails().getEmail())
                .phoneNumber(Optional.ofNullable(entity.getLoginDetails().getPhoneNumber())
                        .map(PhoneNumber::getValue)
                        .orElse(null))
                .isoTwoLetterCountryCode(Optional.ofNullable(entity.getLoginDetails().getPhoneNumber())
                        .map(PhoneNumber::getIsoTwoLetterCountryCode)
                        .orElse(null))
                .username(entity.getLoginDetails().getUsername())
                .credential(Optional.ofNullable(entity.getCredential())
                        .map(credentialDocMapper::map)
                        .orElse(null))
                .role(Optional.ofNullable(entity.getTargetRole())
                        .map(roleDocMapper::map)
                        .orElse(null))
                .deviceId(Optional.ofNullable(entity.getClientInfo())
                        .map(ClientInfo::getDeviceId)
                        .orElse(null))
                .ipV4(Optional.ofNullable(entity.getClientInfo())
                        .map(ClientInfo::getIp)
                        .orElse(null))
                .applicationDetails(Optional.ofNullable(entity.getClientInfo())
                        .map(c->  new AuthSessionDoc.ApplicationDetails(c.getApplication(), c.getPlatform(), c.getVersionName()))
                        .orElse(null))
                .accessToken(Optional.ofNullable(entity.getAccessToken())
                        .map(accessTokenDocMapper::map)
                        .orElse(null))
                .refreshToken(Optional.ofNullable(entity.getRefreshToken())
                        .map(refreshTokenDocMapper::map).orElse(null))
                .version(entity.getVersion())
                .build();
    }
}
