package testing

import io.fluidsonic.graphql.*
import kotlin.test.*


class ParallelVisitorTest {

	@Test
	fun testVisitsADocumentInParallel() {
		val visitorTarget = StackCollectingVisitor.Target()
		val visitors = List(3) {
			StackCollectingVisitor(
				suffix = ".${'A' + it}",
				target = visitorTarget
			)
		}

		val document = GDocument.parse("""
			|query query(${'$'}variable: Int = 1 @foo) @foo(argument: 1) {
			|   field(argument: 1) @foo {
			|      ...fragment @foo
			|      ... on Foo @foo {
			|         field(argument: [{ id: ${'$'}variable }, true, 1, 2.0, "", VALUE, null])
			|      }
			|   }
			|}
		""".trimMargin()).valueWithoutErrorsOrThrow()

		document.accept(visitors.parallelize(), data = StackCollectingVisitor.Data())

		val expectedStacks: List<List<String>> = listOf(
			listOf("Document.A(0)"),
			listOf("Document.A(0)", "Document.B(0)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Name.A(2.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Name.A(2.A)", "Name.B(2.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Name.A(2.A)", "Name.B(2.B)", "Name.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Name.A(3.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Name.A(3.A)", "Name.B(3.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Name.A(3.A)", "Name.B(3.B)", "Name.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "NamedTypeRef.A(3.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "NamedTypeRef.A(3.A)", "NamedTypeRef.B(3.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "NamedTypeRef.A(3.A)", "NamedTypeRef.B(3.B)", "NamedTypeRef.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "NamedTypeRef.A(3.A)", "NamedTypeRef.B(3.B)", "NamedTypeRef.C(3.C)", "Name.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "NamedTypeRef.A(3.A)", "NamedTypeRef.B(3.B)", "NamedTypeRef.C(3.C)", "Name.A(4.A)", "Name.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "NamedTypeRef.A(3.A)", "NamedTypeRef.B(3.B)", "NamedTypeRef.C(3.C)", "Name.A(4.A)", "Name.B(4.B)", "Name.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "IntValue.A(3.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "IntValue.A(3.A)", "IntValue.B(3.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "IntValue.A(3.A)", "IntValue.B(3.B)", "IntValue.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Directive.A(3.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Directive.A(3.A)", "Directive.B(3.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Directive.A(3.A)", "Directive.B(3.B)", "Directive.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Directive.A(3.A)", "Directive.B(3.B)", "Directive.C(3.C)", "Name.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Directive.A(3.A)", "Directive.B(3.B)", "Directive.C(3.C)", "Name.A(4.A)", "Name.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.A(2.A)", "VariableDefinition.B(2.B)", "VariableDefinition.C(2.C)", "Directive.A(3.A)", "Directive.B(3.B)", "Directive.C(3.C)", "Name.A(4.A)", "Name.B(4.B)", "Name.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Name.A(3.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Name.A(3.A)", "Name.B(3.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Name.A(3.A)", "Name.B(3.B)", "Name.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)", "Argument.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)", "Argument.C(3.C)", "Name.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)", "Argument.C(3.C)", "Name.A(4.A)", "Name.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)", "Argument.C(3.C)", "Name.A(4.A)", "Name.B(4.B)", "Name.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)", "Argument.C(3.C)", "IntValue.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)", "Argument.C(3.C)", "IntValue.A(4.A)", "IntValue.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.A(2.A)", "Directive.B(2.B)", "Directive.C(2.C)", "Argument.A(3.A)", "Argument.B(3.B)", "Argument.C(3.C)", "IntValue.A(4.A)", "IntValue.B(4.B)", "IntValue.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Name.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Name.A(4.A)", "Name.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Name.A(4.A)", "Name.B(4.B)", "Name.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)", "Argument.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)", "Argument.C(4.C)", "Name.A(5.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)", "Argument.C(4.C)", "Name.A(5.A)", "Name.B(5.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)", "Argument.C(4.C)", "Name.A(5.A)", "Name.B(5.B)", "Name.C(5.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)", "Argument.C(4.C)", "IntValue.A(5.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)", "Argument.C(4.C)", "IntValue.A(5.A)", "IntValue.B(5.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Argument.A(4.A)", "Argument.B(4.B)", "Argument.C(4.C)", "IntValue.A(5.A)", "IntValue.B(5.B)", "IntValue.C(5.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Directive.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Directive.A(4.A)", "Directive.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Directive.A(4.A)", "Directive.B(4.B)", "Directive.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Directive.A(4.A)", "Directive.B(4.B)", "Directive.C(4.C)", "Name.A(5.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Directive.A(4.A)", "Directive.B(4.B)", "Directive.C(4.C)", "Name.A(5.A)", "Name.B(5.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "Directive.A(4.A)", "Directive.B(4.B)", "Directive.C(4.C)", "Name.A(5.A)", "Name.B(5.B)", "Name.C(5.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Name.A(6.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Name.A(6.A)", "Name.B(6.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Name.A(6.A)", "Name.B(6.B)", "Name.C(6.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Directive.A(6.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)", "Name.A(7.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)", "Name.A(7.A)", "Name.B(7.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "FragmentSelection.A(5.A)", "FragmentSelection.B(5.B)", "FragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)", "Name.A(7.A)", "Name.B(7.B)", "Name.C(7.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "NamedTypeRef.A(6.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "NamedTypeRef.A(6.A)", "NamedTypeRef.B(6.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "NamedTypeRef.A(6.A)", "NamedTypeRef.B(6.B)", "NamedTypeRef.C(6.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "NamedTypeRef.A(6.A)", "NamedTypeRef.B(6.B)", "NamedTypeRef.C(6.C)", "Name.A(7.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "NamedTypeRef.A(6.A)", "NamedTypeRef.B(6.B)", "NamedTypeRef.C(6.C)", "Name.A(7.A)", "Name.B(7.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "NamedTypeRef.A(6.A)", "NamedTypeRef.B(6.B)", "NamedTypeRef.C(6.C)", "Name.A(7.A)", "Name.B(7.B)", "Name.C(7.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "Directive.A(6.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)", "Name.A(7.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)", "Name.A(7.A)", "Name.B(7.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "Directive.A(6.A)", "Directive.B(6.B)", "Directive.C(6.C)", "Name.A(7.A)", "Name.B(7.B)", "Name.C(7.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Name.A(8.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Name.A(8.A)", "Name.B(8.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Name.A(8.A)", "Name.B(8.B)", "Name.C(8.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "Name.A(9.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "Name.A(9.A)", "Name.B(9.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "Name.A(9.A)", "Name.B(9.B)", "Name.C(9.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "Name.A(12.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "Name.A(12.A)", "Name.B(12.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "Name.A(12.A)", "Name.B(12.B)", "Name.C(12.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "VariableRef.A(12.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "VariableRef.A(12.A)", "VariableRef.B(12.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "VariableRef.A(12.A)", "VariableRef.B(12.B)", "VariableRef.C(12.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "VariableRef.A(12.A)", "VariableRef.B(12.B)", "VariableRef.C(12.C)", "Name.A(13.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "VariableRef.A(12.A)", "VariableRef.B(12.B)", "VariableRef.C(12.C)", "Name.A(13.A)", "Name.B(13.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "ObjectValue.A(10.A)", "ObjectValue.B(10.B)", "ObjectValue.C(10.C)", "Argument.A(11.A)", "Argument.B(11.B)", "Argument.C(11.C)", "VariableRef.A(12.A)", "VariableRef.B(12.B)", "VariableRef.C(12.C)", "Name.A(13.A)", "Name.B(13.B)", "Name.C(13.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "BooleanValue.A(10.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "BooleanValue.A(10.A)", "BooleanValue.B(10.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "BooleanValue.A(10.A)", "BooleanValue.B(10.B)", "BooleanValue.C(10.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "IntValue.A(10.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "IntValue.A(10.A)", "IntValue.B(10.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "IntValue.A(10.A)", "IntValue.B(10.B)", "IntValue.C(10.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "FloatValue.A(10.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "FloatValue.A(10.A)", "FloatValue.B(10.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "FloatValue.A(10.A)", "FloatValue.B(10.B)", "FloatValue.C(10.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "StringValue.A(10.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "StringValue.A(10.A)", "StringValue.B(10.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "StringValue.A(10.A)", "StringValue.B(10.B)", "StringValue.C(10.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "EnumValue.A(10.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "EnumValue.A(10.A)", "EnumValue.B(10.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "EnumValue.A(10.A)", "EnumValue.B(10.B)", "EnumValue.C(10.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "NullValue.A(10.A)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "NullValue.A(10.A)", "NullValue.B(10.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.A(1.A)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.A(2.A)", "SelectionSet.B(2.B)", "SelectionSet.C(2.C)", "FieldSelection.A(3.A)", "FieldSelection.B(3.B)", "FieldSelection.C(3.C)", "SelectionSet.A(4.A)", "SelectionSet.B(4.B)", "SelectionSet.C(4.C)", "InlineFragmentSelection.A(5.A)", "InlineFragmentSelection.B(5.B)", "InlineFragmentSelection.C(5.C)", "SelectionSet.A(6.A)", "SelectionSet.B(6.B)", "SelectionSet.C(6.C)", "FieldSelection.A(7.A)", "FieldSelection.B(7.B)", "FieldSelection.C(7.C)", "Argument.A(8.A)", "Argument.B(8.B)", "Argument.C(8.C)", "ListValue.A(9.A)", "ListValue.B(9.B)", "ListValue.C(9.C)", "NullValue.A(10.A)", "NullValue.B(10.B)", "NullValue.C(10.C)")
		)

		val actualStacksString = makePrettyStacks(visitorTarget.stacks)
		val expectedStacksString = makePrettyStacks(expectedStacks)

		assertEquals(expected = expectedStacksString, actual = actualStacksString)
	}


	@Test
	fun testSkipsChildren() {
		val visitorTarget = StackCollectingVisitor.Target()
		val visitors = List(3) { index ->
			StackCollectingVisitor(
				suffix = ".${'A' + index}",
				target = visitorTarget,
				skipsChildrenInNode = {
					when (index) {
						0 -> it is GDocument
						1 -> it is GOperationDefinition
						else -> it is GSelectionSet || it is GVariableDefinition
					}
				}
			)
		}

		val document = GDocument.parse("""
			|query query(${'$'}variable: Int = 1 @foo) @foo(argument: 1) {
			|   field(argument: 1) @foo {
			|      ...fragment @foo
			|      ... on Foo @foo {
			|         field(argument: [{ id: ${'$'}variable }, true, 1, 2.0, "", VALUE, null])
			|      }
			|   }
			|}
		""".trimMargin()).valueWithoutErrorsOrThrow()

		document.accept(visitors.parallelize(), data = StackCollectingVisitor.Data())

		val expectedStacks: List<List<String>> = listOf(
			listOf("Document.A(0)"),
			listOf("Document.A(0)", "Document.B(0)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Name.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Name.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Argument.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Argument.C(3.C)", "Name.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Argument.C(3.C)", "IntValue.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.C(2.C)")
		)

		val actualStacksString = makePrettyStacks(visitorTarget.stacks)
		val expectedStacksString = makePrettyStacks(expectedStacks)

		assertEquals(expected = expectedStacksString, actual = actualStacksString)
	}


	@Test
	fun testAborts() {
		val visitorTarget = StackCollectingVisitor.Target()
		val visitors = List(3) { index ->
			StackCollectingVisitor(
				suffix = ".${'A' + index}",
				target = visitorTarget,
				abortsInNode = {
					when (index) {
						0 -> it is GDocument
						1 -> it is GOperationDefinition
						else -> it is GSelectionSet
					}
				},
				skipsChildrenInNode = {
					index == 2 && it is GVariableDefinition
				}
			)
		}

		val document = GDocument.parse("""
			|query query(${'$'}variable: Int = 1 @foo) @foo(argument: 1) {
			|   field(argument: 1) @foo {
			|      ...fragment @foo
			|      ... on Foo @foo {
			|         field(argument: [{ id: ${'$'}variable }, true, 1, 2.0, "", VALUE, null])
			|      }
			|   }
			|}
		""".trimMargin()).valueWithoutErrorsOrThrow()

		document.accept(visitors.parallelize(), data = StackCollectingVisitor.Data())

		val expectedStacks: List<List<String>> = listOf(
			listOf("Document.A(0)"),
			listOf("Document.A(0)", "Document.B(0)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Name.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "VariableDefinition.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Name.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Argument.C(3.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Argument.C(3.C)", "Name.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "Directive.C(2.C)", "Argument.C(3.C)", "IntValue.C(4.C)"),
			listOf("Document.A(0)", "Document.B(0)", "Document.C(0)", "OperationDefinition.B(1.B)", "OperationDefinition.C(1.C)", "SelectionSet.C(2.C)")
		)

		val actualStacksString = makePrettyStacks(visitorTarget.stacks)
		val expectedStacksString = makePrettyStacks(expectedStacks)

		assertEquals(expected = expectedStacksString, actual = actualStacksString)
	}


	private fun makePrettyStacks(stacks: List<List<String>>) =
		"\n" + stacks.joinToString(",\n") { stack ->
			"listOf(" + stack.joinToString(", ") { "\"$it\"" } + ")"
		} + "\n"
}
