package io.fluidsonic.country


public class Country private constructor(
	public val code: CountryCode,
	private val code_iso3116_alpha3: String,
	public val name: String,
	public val numericCode: Int,
) {

	init {
		freeze()
	}


	public fun code(format: CountryCode.Format): String =
		when (format) {
			CountryCode.Format.iso3166_alpha2 -> code.toString()
			CountryCode.Format.iso3166_alpha3 -> code_iso3116_alpha3
		}


	override fun toString(): String =
		code.toString()


	public companion object {

		// https://www.iso.org/obp/ui/
		public val all: Set<Country> = hashSetOf(
			make(code = "AD", code_iso3116_alpha3 = "AND", numericCode = 20, name = "Andorra"),
			make(code = "AE", code_iso3116_alpha3 = "ARE", numericCode = 784, name = "United Arab Emirates (the)"),
			make(code = "AF", code_iso3116_alpha3 = "AFG", numericCode = 4, name = "Afghanistan"),
			make(code = "AG", code_iso3116_alpha3 = "ATG", numericCode = 28, name = "Antigua and Barbuda"),
			make(code = "AI", code_iso3116_alpha3 = "AIA", numericCode = 660, name = "Anguilla"),
			make(code = "AL", code_iso3116_alpha3 = "ALB", numericCode = 8, name = "Albania"),
			make(code = "AM", code_iso3116_alpha3 = "ARM", numericCode = 51, name = "Armenia"),
			make(code = "AO", code_iso3116_alpha3 = "AGO", numericCode = 24, name = "Angola"),
			make(code = "AQ", code_iso3116_alpha3 = "ATA", numericCode = 10, name = "Antarctica"),
			make(code = "AR", code_iso3116_alpha3 = "ARG", numericCode = 32, name = "Argentina"),
			make(code = "AS", code_iso3116_alpha3 = "ASM", numericCode = 16, name = "American Samoa"),
			make(code = "AT", code_iso3116_alpha3 = "AUT", numericCode = 40, name = "Austria"),
			make(code = "AU", code_iso3116_alpha3 = "AUS", numericCode = 36, name = "Australia"),
			make(code = "AW", code_iso3116_alpha3 = "ABW", numericCode = 533, name = "Aruba"),
			make(code = "AX", code_iso3116_alpha3 = "ALA", numericCode = 248, name = "Åland Islands"),
			make(code = "AZ", code_iso3116_alpha3 = "AZE", numericCode = 31, name = "Azerbaijan"),
			make(code = "BA", code_iso3116_alpha3 = "BIH", numericCode = 70, name = "Bosnia and Herzegovina"),
			make(code = "BB", code_iso3116_alpha3 = "BRB", numericCode = 52, name = "Barbados"),
			make(code = "BD", code_iso3116_alpha3 = "BGD", numericCode = 50, name = "Bangladesh"),
			make(code = "BE", code_iso3116_alpha3 = "BEL", numericCode = 56, name = "Belgium"),
			make(code = "BF", code_iso3116_alpha3 = "BFA", numericCode = 854, name = "Burkina Faso"),
			make(code = "BG", code_iso3116_alpha3 = "BGR", numericCode = 100, name = "Bulgaria"),
			make(code = "BH", code_iso3116_alpha3 = "BHR", numericCode = 48, name = "Bahrain"),
			make(code = "BI", code_iso3116_alpha3 = "BDI", numericCode = 108, name = "Burundi"),
			make(code = "BJ", code_iso3116_alpha3 = "BEN", numericCode = 204, name = "Benin"),
			make(code = "BL", code_iso3116_alpha3 = "BLM", numericCode = 652, name = "Saint Barthélemy"),
			make(code = "BM", code_iso3116_alpha3 = "BMU", numericCode = 60, name = "Bermuda"),
			make(code = "BN", code_iso3116_alpha3 = "BRN", numericCode = 96, name = "Brunei Darussalam"),
			make(code = "BO", code_iso3116_alpha3 = "BOL", numericCode = 68, name = "Bolivia (Plurinational State of)"),
			make(code = "BQ", code_iso3116_alpha3 = "BES", numericCode = 535, name = "Bonaire, Sint Eustatius and Saba"),
			make(code = "BR", code_iso3116_alpha3 = "BRA", numericCode = 76, name = "Brazil"),
			make(code = "BS", code_iso3116_alpha3 = "BHS", numericCode = 44, name = "Bahamas (the)"),
			make(code = "BT", code_iso3116_alpha3 = "BTN", numericCode = 64, name = "Bhutan"),
			make(code = "BV", code_iso3116_alpha3 = "BVT", numericCode = 74, name = "Bouvet Island"),
			make(code = "BW", code_iso3116_alpha3 = "BWA", numericCode = 72, name = "Botswana"),
			make(code = "BY", code_iso3116_alpha3 = "BLR", numericCode = 112, name = "Belarus"),
			make(code = "BZ", code_iso3116_alpha3 = "BLZ", numericCode = 84, name = "Belize"),
			make(code = "CA", code_iso3116_alpha3 = "CAN", numericCode = 124, name = "Canada"),
			make(code = "CC", code_iso3116_alpha3 = "CCK", numericCode = 166, name = "Cocos (Keeling) Islands (the)"),
			make(code = "CD", code_iso3116_alpha3 = "COD", numericCode = 180, name = "Congo (the Democratic Republic of the)"),
			make(code = "CF", code_iso3116_alpha3 = "CAF", numericCode = 140, name = "Central African Republic (the)"),
			make(code = "CG", code_iso3116_alpha3 = "COG", numericCode = 178, name = "Congo (the)"),
			make(code = "CH", code_iso3116_alpha3 = "CHE", numericCode = 756, name = "Switzerland"),
			make(code = "CI", code_iso3116_alpha3 = "CIV", numericCode = 384, name = "Côte d'Ivoire"),
			make(code = "CK", code_iso3116_alpha3 = "COK", numericCode = 184, name = "Cook Islands (the)"),
			make(code = "CL", code_iso3116_alpha3 = "CHL", numericCode = 152, name = "Chile"),
			make(code = "CM", code_iso3116_alpha3 = "CMR", numericCode = 120, name = "Cameroon"),
			make(code = "CN", code_iso3116_alpha3 = "CHN", numericCode = 156, name = "China"),
			make(code = "CO", code_iso3116_alpha3 = "COL", numericCode = 170, name = "Colombia"),
			make(code = "CR", code_iso3116_alpha3 = "CRI", numericCode = 188, name = "Costa Rica"),
			make(code = "CU", code_iso3116_alpha3 = "CUB", numericCode = 192, name = "Cuba"),
			make(code = "CV", code_iso3116_alpha3 = "CPV", numericCode = 132, name = "Cabo Verde"),
			make(code = "CW", code_iso3116_alpha3 = "CUW", numericCode = 531, name = "Curaçao"),
			make(code = "CX", code_iso3116_alpha3 = "CXR", numericCode = 162, name = "Christmas Island"),
			make(code = "CY", code_iso3116_alpha3 = "CYP", numericCode = 196, name = "Cyprus"),
			make(code = "CZ", code_iso3116_alpha3 = "CZE", numericCode = 203, name = "Czechia"),
			make(code = "DE", code_iso3116_alpha3 = "DEU", numericCode = 276, name = "Germany"),
			make(code = "DJ", code_iso3116_alpha3 = "DJI", numericCode = 262, name = "Djibouti"),
			make(code = "DK", code_iso3116_alpha3 = "DNK", numericCode = 208, name = "Denmark"),
			make(code = "DM", code_iso3116_alpha3 = "DMA", numericCode = 212, name = "Dominica"),
			make(code = "DO", code_iso3116_alpha3 = "DOM", numericCode = 214, name = "Dominican Republic (the)"),
			make(code = "DZ", code_iso3116_alpha3 = "DZA", numericCode = 12, name = "Algeria"),
			make(code = "EC", code_iso3116_alpha3 = "ECU", numericCode = 218, name = "Ecuador"),
			make(code = "EE", code_iso3116_alpha3 = "EST", numericCode = 233, name = "Estonia"),
			make(code = "EG", code_iso3116_alpha3 = "EGY", numericCode = 818, name = "Egypt"),
			make(code = "EH", code_iso3116_alpha3 = "ESH", numericCode = 732, name = "Western Sahara*"),
			make(code = "ER", code_iso3116_alpha3 = "ERI", numericCode = 232, name = "Eritrea"),
			make(code = "ES", code_iso3116_alpha3 = "ESP", numericCode = 724, name = "Spain"),
			make(code = "ET", code_iso3116_alpha3 = "ETH", numericCode = 231, name = "Ethiopia"),
			make(code = "FI", code_iso3116_alpha3 = "FIN", numericCode = 246, name = "Finland"),
			make(code = "FJ", code_iso3116_alpha3 = "FJI", numericCode = 242, name = "Fiji"),
			make(code = "FK", code_iso3116_alpha3 = "FLK", numericCode = 238, name = "Falkland Islands (the) [Malvinas]"),
			make(code = "FM", code_iso3116_alpha3 = "FSM", numericCode = 583, name = "Micronesia (Federated States of)"),
			make(code = "FO", code_iso3116_alpha3 = "FRO", numericCode = 234, name = "Faroe Islands (the)"),
			make(code = "FR", code_iso3116_alpha3 = "FRA", numericCode = 250, name = "France"),
			make(code = "GA", code_iso3116_alpha3 = "GAB", numericCode = 266, name = "Gabon"),
			make(code = "GB", code_iso3116_alpha3 = "GBR", numericCode = 826, name = "United Kingdom of Great Britain and Northern Ireland (the)"),
			make(code = "GD", code_iso3116_alpha3 = "GRD", numericCode = 308, name = "Grenada"),
			make(code = "GE", code_iso3116_alpha3 = "GEO", numericCode = 268, name = "Georgia"),
			make(code = "GF", code_iso3116_alpha3 = "GUF", numericCode = 254, name = "French Guiana"),
			make(code = "GG", code_iso3116_alpha3 = "GGY", numericCode = 831, name = "Guernsey"),
			make(code = "GH", code_iso3116_alpha3 = "GHA", numericCode = 288, name = "Ghana"),
			make(code = "GI", code_iso3116_alpha3 = "GIB", numericCode = 292, name = "Gibraltar"),
			make(code = "GL", code_iso3116_alpha3 = "GRL", numericCode = 304, name = "Greenland"),
			make(code = "GM", code_iso3116_alpha3 = "GMB", numericCode = 270, name = "Gambia (the)"),
			make(code = "GN", code_iso3116_alpha3 = "GIN", numericCode = 324, name = "Guinea"),
			make(code = "GP", code_iso3116_alpha3 = "GLP", numericCode = 312, name = "Guadeloupe"),
			make(code = "GQ", code_iso3116_alpha3 = "GNQ", numericCode = 226, name = "Equatorial Guinea"),
			make(code = "GR", code_iso3116_alpha3 = "GRC", numericCode = 300, name = "Greece"),
			make(code = "GS", code_iso3116_alpha3 = "SGS", numericCode = 239, name = "South Georgia and the South Sandwich Islands"),
			make(code = "GT", code_iso3116_alpha3 = "GTM", numericCode = 320, name = "Guatemala"),
			make(code = "GU", code_iso3116_alpha3 = "GUM", numericCode = 316, name = "Guam"),
			make(code = "GW", code_iso3116_alpha3 = "GNB", numericCode = 624, name = "Guinea-Bissau"),
			make(code = "GY", code_iso3116_alpha3 = "GUY", numericCode = 328, name = "Guyana"),
			make(code = "HK", code_iso3116_alpha3 = "HKG", numericCode = 344, name = "Hong Kong"),
			make(code = "HM", code_iso3116_alpha3 = "HMD", numericCode = 334, name = "Heard Island and McDonald Islands"),
			make(code = "HN", code_iso3116_alpha3 = "HND", numericCode = 340, name = "Honduras"),
			make(code = "HR", code_iso3116_alpha3 = "HRV", numericCode = 191, name = "Croatia"),
			make(code = "HT", code_iso3116_alpha3 = "HTI", numericCode = 332, name = "Haiti"),
			make(code = "HU", code_iso3116_alpha3 = "HUN", numericCode = 348, name = "Hungary"),
			make(code = "ID", code_iso3116_alpha3 = "IDN", numericCode = 360, name = "Indonesia"),
			make(code = "IE", code_iso3116_alpha3 = "IRL", numericCode = 372, name = "Ireland"),
			make(code = "IL", code_iso3116_alpha3 = "ISR", numericCode = 376, name = "Israel"),
			make(code = "IM", code_iso3116_alpha3 = "IMN", numericCode = 833, name = "Isle of Man"),
			make(code = "IN", code_iso3116_alpha3 = "IND", numericCode = 356, name = "India"),
			make(code = "IO", code_iso3116_alpha3 = "IOT", numericCode = 86, name = "British Indian Ocean Territory (the)"),
			make(code = "IQ", code_iso3116_alpha3 = "IRQ", numericCode = 368, name = "Iraq"),
			make(code = "IR", code_iso3116_alpha3 = "IRN", numericCode = 364, name = "Iran (Islamic Republic of)"),
			make(code = "IS", code_iso3116_alpha3 = "ISL", numericCode = 352, name = "Iceland"),
			make(code = "IT", code_iso3116_alpha3 = "ITA", numericCode = 380, name = "Italy"),
			make(code = "JE", code_iso3116_alpha3 = "JEY", numericCode = 832, name = "Jersey"),
			make(code = "JM", code_iso3116_alpha3 = "JAM", numericCode = 388, name = "Jamaica"),
			make(code = "JO", code_iso3116_alpha3 = "JOR", numericCode = 400, name = "Jordan"),
			make(code = "JP", code_iso3116_alpha3 = "JPN", numericCode = 392, name = "Japan"),
			make(code = "KE", code_iso3116_alpha3 = "KEN", numericCode = 404, name = "Kenya"),
			make(code = "KG", code_iso3116_alpha3 = "KGZ", numericCode = 417, name = "Kyrgyzstan"),
			make(code = "KH", code_iso3116_alpha3 = "KHM", numericCode = 116, name = "Cambodia"),
			make(code = "KI", code_iso3116_alpha3 = "KIR", numericCode = 296, name = "Kiribati"),
			make(code = "KM", code_iso3116_alpha3 = "COM", numericCode = 174, name = "Comoros (the)"),
			make(code = "KN", code_iso3116_alpha3 = "KNA", numericCode = 659, name = "Saint Kitts and Nevis"),
			make(code = "KP", code_iso3116_alpha3 = "PRK", numericCode = 408, name = "Korea (the Democratic People's Republic of)"),
			make(code = "KR", code_iso3116_alpha3 = "KOR", numericCode = 410, name = "Korea (the Republic of)"),
			make(code = "KW", code_iso3116_alpha3 = "KWT", numericCode = 414, name = "Kuwait"),
			make(code = "KY", code_iso3116_alpha3 = "CYM", numericCode = 136, name = "Cayman Islands (the)"),
			make(code = "KZ", code_iso3116_alpha3 = "KAZ", numericCode = 398, name = "Kazakhstan"),
			make(code = "LA", code_iso3116_alpha3 = "LAO", numericCode = 418, name = "Lao People's Democratic Republic (the)"),
			make(code = "LB", code_iso3116_alpha3 = "LBN", numericCode = 422, name = "Lebanon"),
			make(code = "LC", code_iso3116_alpha3 = "LCA", numericCode = 662, name = "Saint Lucia"),
			make(code = "LI", code_iso3116_alpha3 = "LIE", numericCode = 438, name = "Liechtenstein"),
			make(code = "LK", code_iso3116_alpha3 = "LKA", numericCode = 144, name = "Sri Lanka"),
			make(code = "LR", code_iso3116_alpha3 = "LBR", numericCode = 430, name = "Liberia"),
			make(code = "LS", code_iso3116_alpha3 = "LSO", numericCode = 426, name = "Lesotho"),
			make(code = "LT", code_iso3116_alpha3 = "LTU", numericCode = 440, name = "Lithuania"),
			make(code = "LU", code_iso3116_alpha3 = "LUX", numericCode = 442, name = "Luxembourg"),
			make(code = "LV", code_iso3116_alpha3 = "LVA", numericCode = 428, name = "Latvia"),
			make(code = "LY", code_iso3116_alpha3 = "LBY", numericCode = 434, name = "Libya"),
			make(code = "MA", code_iso3116_alpha3 = "MAR", numericCode = 504, name = "Morocco"),
			make(code = "MC", code_iso3116_alpha3 = "MCO", numericCode = 492, name = "Monaco"),
			make(code = "MD", code_iso3116_alpha3 = "MDA", numericCode = 498, name = "Moldova (the Republic of)"),
			make(code = "ME", code_iso3116_alpha3 = "MNE", numericCode = 499, name = "Montenegro"),
			make(code = "MF", code_iso3116_alpha3 = "MAF", numericCode = 663, name = "Saint Martin (French part)"),
			make(code = "MG", code_iso3116_alpha3 = "MDG", numericCode = 450, name = "Madagascar"),
			make(code = "MH", code_iso3116_alpha3 = "MHL", numericCode = 584, name = "Marshall Islands (the)"),
			make(code = "MK", code_iso3116_alpha3 = "MKD", numericCode = 807, name = "North Macedonia"),
			make(code = "ML", code_iso3116_alpha3 = "MLI", numericCode = 466, name = "Mali"),
			make(code = "MM", code_iso3116_alpha3 = "MMR", numericCode = 104, name = "Myanmar"),
			make(code = "MN", code_iso3116_alpha3 = "MNG", numericCode = 496, name = "Mongolia"),
			make(code = "MO", code_iso3116_alpha3 = "MAC", numericCode = 446, name = "Macao"),
			make(code = "MP", code_iso3116_alpha3 = "MNP", numericCode = 580, name = "Northern Mariana Islands (the)"),
			make(code = "MQ", code_iso3116_alpha3 = "MTQ", numericCode = 474, name = "Martinique"),
			make(code = "MR", code_iso3116_alpha3 = "MRT", numericCode = 478, name = "Mauritania"),
			make(code = "MS", code_iso3116_alpha3 = "MSR", numericCode = 500, name = "Montserrat"),
			make(code = "MT", code_iso3116_alpha3 = "MLT", numericCode = 470, name = "Malta"),
			make(code = "MU", code_iso3116_alpha3 = "MUS", numericCode = 480, name = "Mauritius"),
			make(code = "MV", code_iso3116_alpha3 = "MDV", numericCode = 462, name = "Maldives"),
			make(code = "MW", code_iso3116_alpha3 = "MWI", numericCode = 454, name = "Malawi"),
			make(code = "MX", code_iso3116_alpha3 = "MEX", numericCode = 484, name = "Mexico"),
			make(code = "MY", code_iso3116_alpha3 = "MYS", numericCode = 458, name = "Malaysia"),
			make(code = "MZ", code_iso3116_alpha3 = "MOZ", numericCode = 508, name = "Mozambique"),
			make(code = "NA", code_iso3116_alpha3 = "NAM", numericCode = 516, name = "Namibia"),
			make(code = "NC", code_iso3116_alpha3 = "NCL", numericCode = 540, name = "New Caledonia"),
			make(code = "NE", code_iso3116_alpha3 = "NER", numericCode = 562, name = "Niger (the)"),
			make(code = "NF", code_iso3116_alpha3 = "NFK", numericCode = 574, name = "Norfolk Island"),
			make(code = "NG", code_iso3116_alpha3 = "NGA", numericCode = 566, name = "Nigeria"),
			make(code = "NI", code_iso3116_alpha3 = "NIC", numericCode = 558, name = "Nicaragua"),
			make(code = "NL", code_iso3116_alpha3 = "NLD", numericCode = 528, name = "Netherlands (the)"),
			make(code = "NO", code_iso3116_alpha3 = "NOR", numericCode = 578, name = "Norway"),
			make(code = "NP", code_iso3116_alpha3 = "NPL", numericCode = 524, name = "Nepal"),
			make(code = "NR", code_iso3116_alpha3 = "NRU", numericCode = 520, name = "Nauru"),
			make(code = "NU", code_iso3116_alpha3 = "NIU", numericCode = 570, name = "Niue"),
			make(code = "NZ", code_iso3116_alpha3 = "NZL", numericCode = 554, name = "New Zealand"),
			make(code = "OM", code_iso3116_alpha3 = "OMN", numericCode = 512, name = "Oman"),
			make(code = "PA", code_iso3116_alpha3 = "PAN", numericCode = 591, name = "Panama"),
			make(code = "PE", code_iso3116_alpha3 = "PER", numericCode = 604, name = "Peru"),
			make(code = "PF", code_iso3116_alpha3 = "PYF", numericCode = 258, name = "French Polynesia"),
			make(code = "PG", code_iso3116_alpha3 = "PNG", numericCode = 598, name = "Papua New Guinea"),
			make(code = "PH", code_iso3116_alpha3 = "PHL", numericCode = 608, name = "Philippines (the)"),
			make(code = "PK", code_iso3116_alpha3 = "PAK", numericCode = 586, name = "Pakistan"),
			make(code = "PL", code_iso3116_alpha3 = "POL", numericCode = 616, name = "Poland"),
			make(code = "PM", code_iso3116_alpha3 = "SPM", numericCode = 666, name = "Saint Pierre and Miquelon"),
			make(code = "PN", code_iso3116_alpha3 = "PCN", numericCode = 612, name = "Pitcairn"),
			make(code = "PR", code_iso3116_alpha3 = "PRI", numericCode = 630, name = "Puerto Rico"),
			make(code = "PS", code_iso3116_alpha3 = "PSE", numericCode = 275, name = "Palestine, State of"),
			make(code = "PT", code_iso3116_alpha3 = "PRT", numericCode = 620, name = "Portugal"),
			make(code = "PW", code_iso3116_alpha3 = "PLW", numericCode = 585, name = "Palau"),
			make(code = "PY", code_iso3116_alpha3 = "PRY", numericCode = 600, name = "Paraguay"),
			make(code = "QA", code_iso3116_alpha3 = "QAT", numericCode = 634, name = "Qatar"),
			make(code = "RE", code_iso3116_alpha3 = "REU", numericCode = 638, name = "Réunion"),
			make(code = "RO", code_iso3116_alpha3 = "ROU", numericCode = 642, name = "Romania"),
			make(code = "RS", code_iso3116_alpha3 = "SRB", numericCode = 688, name = "Serbia"),
			make(code = "RU", code_iso3116_alpha3 = "RUS", numericCode = 643, name = "Russian Federation (the)"),
			make(code = "RW", code_iso3116_alpha3 = "RWA", numericCode = 646, name = "Rwanda"),
			make(code = "SA", code_iso3116_alpha3 = "SAU", numericCode = 682, name = "Saudi Arabia"),
			make(code = "SB", code_iso3116_alpha3 = "SLB", numericCode = 90, name = "Solomon Islands"),
			make(code = "SC", code_iso3116_alpha3 = "SYC", numericCode = 690, name = "Seychelles"),
			make(code = "SD", code_iso3116_alpha3 = "SDN", numericCode = 729, name = "Sudan (the)"),
			make(code = "SE", code_iso3116_alpha3 = "SWE", numericCode = 752, name = "Sweden"),
			make(code = "SG", code_iso3116_alpha3 = "SGP", numericCode = 702, name = "Singapore"),
			make(code = "SH", code_iso3116_alpha3 = "SHN", numericCode = 654, name = "Saint Helena, Ascension and Tristan da Cunha"),
			make(code = "SI", code_iso3116_alpha3 = "SVN", numericCode = 705, name = "Slovenia"),
			make(code = "SJ", code_iso3116_alpha3 = "SJM", numericCode = 744, name = "Svalbard and Jan Mayen"),
			make(code = "SK", code_iso3116_alpha3 = "SVK", numericCode = 703, name = "Slovakia"),
			make(code = "SL", code_iso3116_alpha3 = "SLE", numericCode = 694, name = "Sierra Leone"),
			make(code = "SM", code_iso3116_alpha3 = "SMR", numericCode = 674, name = "San Marino"),
			make(code = "SN", code_iso3116_alpha3 = "SEN", numericCode = 686, name = "Senegal"),
			make(code = "SO", code_iso3116_alpha3 = "SOM", numericCode = 706, name = "Somalia"),
			make(code = "SR", code_iso3116_alpha3 = "SUR", numericCode = 740, name = "Suriname"),
			make(code = "SS", code_iso3116_alpha3 = "SSD", numericCode = 728, name = "South Sudan"),
			make(code = "ST", code_iso3116_alpha3 = "STP", numericCode = 678, name = "Sao Tome and Principe"),
			make(code = "SV", code_iso3116_alpha3 = "SLV", numericCode = 222, name = "El Salvador"),
			make(code = "SX", code_iso3116_alpha3 = "SXM", numericCode = 534, name = "Sint Maarten (Dutch part)"),
			make(code = "SY", code_iso3116_alpha3 = "SYR", numericCode = 760, name = "Syrian Arab Republic (the)"),
			make(code = "SZ", code_iso3116_alpha3 = "SWZ", numericCode = 748, name = "Eswatini"),
			make(code = "TC", code_iso3116_alpha3 = "TCA", numericCode = 796, name = "Turks and Caicos Islands (the)"),
			make(code = "TD", code_iso3116_alpha3 = "TCD", numericCode = 148, name = "Chad"),
			make(code = "TF", code_iso3116_alpha3 = "ATF", numericCode = 260, name = "French Southern Territories (the)"),
			make(code = "TG", code_iso3116_alpha3 = "TGO", numericCode = 768, name = "Togo"),
			make(code = "TH", code_iso3116_alpha3 = "THA", numericCode = 764, name = "Thailand"),
			make(code = "TJ", code_iso3116_alpha3 = "TJK", numericCode = 762, name = "Tajikistan"),
			make(code = "TK", code_iso3116_alpha3 = "TKL", numericCode = 772, name = "Tokelau"),
			make(code = "TL", code_iso3116_alpha3 = "TLS", numericCode = 626, name = "Timor-Leste"),
			make(code = "TM", code_iso3116_alpha3 = "TKM", numericCode = 795, name = "Turkmenistan"),
			make(code = "TN", code_iso3116_alpha3 = "TUN", numericCode = 788, name = "Tunisia"),
			make(code = "TO", code_iso3116_alpha3 = "TON", numericCode = 776, name = "Tonga"),
			make(code = "TR", code_iso3116_alpha3 = "TUR", numericCode = 792, name = "Turkey"),
			make(code = "TT", code_iso3116_alpha3 = "TTO", numericCode = 780, name = "Trinidad and Tobago"),
			make(code = "TV", code_iso3116_alpha3 = "TUV", numericCode = 798, name = "Tuvalu"),
			make(code = "TW", code_iso3116_alpha3 = "TWN", numericCode = 158, name = "Taiwan (Province of China)"),
			make(code = "TZ", code_iso3116_alpha3 = "TZA", numericCode = 834, name = "Tanzania, the United Republic of"),
			make(code = "UA", code_iso3116_alpha3 = "UKR", numericCode = 804, name = "Ukraine"),
			make(code = "UG", code_iso3116_alpha3 = "UGA", numericCode = 800, name = "Uganda"),
			make(code = "UM", code_iso3116_alpha3 = "UMI", numericCode = 581, name = "United States Minor Outlying Islands (the)"),
			make(code = "US", code_iso3116_alpha3 = "USA", numericCode = 840, name = "United States of America (the)"),
			make(code = "UY", code_iso3116_alpha3 = "URY", numericCode = 858, name = "Uruguay"),
			make(code = "UZ", code_iso3116_alpha3 = "UZB", numericCode = 860, name = "Uzbekistan"),
			make(code = "VA", code_iso3116_alpha3 = "VAT", numericCode = 336, name = "Holy See (the)"),
			make(code = "VC", code_iso3116_alpha3 = "VCT", numericCode = 670, name = "Saint Vincent and the Grenadines"),
			make(code = "VE", code_iso3116_alpha3 = "VEN", numericCode = 862, name = "Venezuela (Bolivarian Republic of)"),
			make(code = "VG", code_iso3116_alpha3 = "VGB", numericCode = 92, name = "Virgin Islands (British)"),
			make(code = "VI", code_iso3116_alpha3 = "VIR", numericCode = 850, name = "Virgin Islands (U.S.)"),
			make(code = "VN", code_iso3116_alpha3 = "VNM", numericCode = 704, name = "Viet Nam"),
			make(code = "VU", code_iso3116_alpha3 = "VUT", numericCode = 548, name = "Vanuatu"),
			make(code = "WF", code_iso3116_alpha3 = "WLF", numericCode = 876, name = "Wallis and Futuna"),
			make(code = "WS", code_iso3116_alpha3 = "WSM", numericCode = 882, name = "Samoa"),
			make(code = "YE", code_iso3116_alpha3 = "YEM", numericCode = 887, name = "Yemen"),
			make(code = "YT", code_iso3116_alpha3 = "MYT", numericCode = 175, name = "Mayotte"),
			make(code = "ZA", code_iso3116_alpha3 = "ZAF", numericCode = 710, name = "South Africa"),
			make(code = "ZM", code_iso3116_alpha3 = "ZMB", numericCode = 894, name = "Zambia"),
			make(code = "ZW", code_iso3116_alpha3 = "ZWE", numericCode = 716, name = "Zimbabwe"),
		)


		private val allByCode: Map<CountryCode, Country> = all.associateByTo(hashMapOf()) { it.code }


		private fun make(code: String, code_iso3116_alpha3: String, numericCode: Int, name: String): Country =
			Country(
				code = CountryCode(code),
				code_iso3116_alpha3 = code_iso3116_alpha3,
				name = name,
				numericCode = numericCode
			)


		public fun forCode(code: CountryCode): Country =
			forCodeOrNull(code) ?: error("Invalid ISO 3166-1 alpha-2 country code: $code")


		public fun forCodeOrNull(code: CountryCode): Country? =
			allByCode[code]


		public fun forCode(code: String): Country =
			forCodeOrNull(CountryCode.parse(code)) ?: error("Invalid ISO 3166-1 alpha-2 country code: $code")


		public fun forCodeOrNull(code: String): Country? =
			CountryCode.parseOrNull(code)?.let(::forCodeOrNull)
	}
}
