package io.fluidsonic.country


public class Country private constructor(
	public val code: CountryCode,
	private val code_iso3116_alpha3: String,
	public val numericCode: Int,
) {

	init {
		freeze()
	}


	public fun code(format: CountryCode.Format): String =
		when (format) {
			CountryCode.Format.iso3166_alpha2 -> code.toString()
			CountryCode.Format.iso3166_alpha3 -> code_iso3116_alpha3
		}


	override fun toString(): String =
		code.toString()


	public companion object {

		// https://www.iso.org/obp/ui/
		public val all: Set<Country> = hashSetOf(
			make(code = "AD", code_iso3116_alpha3 = "AND", numericCode = 20),
			make(code = "AE", code_iso3116_alpha3 = "ARE", numericCode = 784),
			make(code = "AF", code_iso3116_alpha3 = "AFG", numericCode = 4),
			make(code = "AG", code_iso3116_alpha3 = "ATG", numericCode = 28),
			make(code = "AI", code_iso3116_alpha3 = "AIA", numericCode = 660),
			make(code = "AL", code_iso3116_alpha3 = "ALB", numericCode = 8),
			make(code = "AM", code_iso3116_alpha3 = "ARM", numericCode = 51),
			make(code = "AO", code_iso3116_alpha3 = "AGO", numericCode = 24),
			make(code = "AQ", code_iso3116_alpha3 = "ATA", numericCode = 10),
			make(code = "AR", code_iso3116_alpha3 = "ARG", numericCode = 32),
			make(code = "AS", code_iso3116_alpha3 = "ASM", numericCode = 16),
			make(code = "AT", code_iso3116_alpha3 = "AUT", numericCode = 40),
			make(code = "AU", code_iso3116_alpha3 = "AUS", numericCode = 36),
			make(code = "AW", code_iso3116_alpha3 = "ABW", numericCode = 533),
			make(code = "AX", code_iso3116_alpha3 = "ALA", numericCode = 248),
			make(code = "AZ", code_iso3116_alpha3 = "AZE", numericCode = 31),
			make(code = "BA", code_iso3116_alpha3 = "BIH", numericCode = 70),
			make(code = "BB", code_iso3116_alpha3 = "BRB", numericCode = 52),
			make(code = "BD", code_iso3116_alpha3 = "BGD", numericCode = 50),
			make(code = "BE", code_iso3116_alpha3 = "BEL", numericCode = 56),
			make(code = "BF", code_iso3116_alpha3 = "BFA", numericCode = 854),
			make(code = "BG", code_iso3116_alpha3 = "BGR", numericCode = 100),
			make(code = "BH", code_iso3116_alpha3 = "BHR", numericCode = 48),
			make(code = "BI", code_iso3116_alpha3 = "BDI", numericCode = 108),
			make(code = "BJ", code_iso3116_alpha3 = "BEN", numericCode = 204),
			make(code = "BL", code_iso3116_alpha3 = "BLM", numericCode = 652),
			make(code = "BM", code_iso3116_alpha3 = "BMU", numericCode = 60),
			make(code = "BN", code_iso3116_alpha3 = "BRN", numericCode = 96),
			make(code = "BO", code_iso3116_alpha3 = "BOL", numericCode = 68),
			make(code = "BQ", code_iso3116_alpha3 = "BES", numericCode = 535),
			make(code = "BR", code_iso3116_alpha3 = "BRA", numericCode = 76),
			make(code = "BS", code_iso3116_alpha3 = "BHS", numericCode = 44),
			make(code = "BT", code_iso3116_alpha3 = "BTN", numericCode = 64),
			make(code = "BV", code_iso3116_alpha3 = "BVT", numericCode = 74),
			make(code = "BW", code_iso3116_alpha3 = "BWA", numericCode = 72),
			make(code = "BY", code_iso3116_alpha3 = "BLR", numericCode = 112),
			make(code = "BZ", code_iso3116_alpha3 = "BLZ", numericCode = 84),
			make(code = "CA", code_iso3116_alpha3 = "CAN", numericCode = 124),
			make(code = "CC", code_iso3116_alpha3 = "CCK", numericCode = 166),
			make(code = "CD", code_iso3116_alpha3 = "COD", numericCode = 180),
			make(code = "CF", code_iso3116_alpha3 = "CAF", numericCode = 140),
			make(code = "CG", code_iso3116_alpha3 = "COG", numericCode = 178),
			make(code = "CH", code_iso3116_alpha3 = "CHE", numericCode = 756),
			make(code = "CI", code_iso3116_alpha3 = "CIV", numericCode = 384),
			make(code = "CK", code_iso3116_alpha3 = "COK", numericCode = 184),
			make(code = "CL", code_iso3116_alpha3 = "CHL", numericCode = 152),
			make(code = "CM", code_iso3116_alpha3 = "CMR", numericCode = 120),
			make(code = "CN", code_iso3116_alpha3 = "CHN", numericCode = 156),
			make(code = "CO", code_iso3116_alpha3 = "COL", numericCode = 170),
			make(code = "CR", code_iso3116_alpha3 = "CRI", numericCode = 188),
			make(code = "CU", code_iso3116_alpha3 = "CUB", numericCode = 192),
			make(code = "CV", code_iso3116_alpha3 = "CPV", numericCode = 132),
			make(code = "CW", code_iso3116_alpha3 = "CUW", numericCode = 531),
			make(code = "CX", code_iso3116_alpha3 = "CXR", numericCode = 162),
			make(code = "CY", code_iso3116_alpha3 = "CYP", numericCode = 196),
			make(code = "CZ", code_iso3116_alpha3 = "CZE", numericCode = 203),
			make(code = "DE", code_iso3116_alpha3 = "DEU", numericCode = 276),
			make(code = "DJ", code_iso3116_alpha3 = "DJI", numericCode = 262),
			make(code = "DK", code_iso3116_alpha3 = "DNK", numericCode = 208),
			make(code = "DM", code_iso3116_alpha3 = "DMA", numericCode = 212),
			make(code = "DO", code_iso3116_alpha3 = "DOM", numericCode = 214),
			make(code = "DZ", code_iso3116_alpha3 = "DZA", numericCode = 12),
			make(code = "EC", code_iso3116_alpha3 = "ECU", numericCode = 218),
			make(code = "EE", code_iso3116_alpha3 = "EST", numericCode = 233),
			make(code = "EG", code_iso3116_alpha3 = "EGY", numericCode = 818),
			make(code = "EH", code_iso3116_alpha3 = "ESH", numericCode = 732),
			make(code = "ER", code_iso3116_alpha3 = "ERI", numericCode = 232),
			make(code = "ES", code_iso3116_alpha3 = "ESP", numericCode = 724),
			make(code = "ET", code_iso3116_alpha3 = "ETH", numericCode = 231),
			make(code = "FI", code_iso3116_alpha3 = "FIN", numericCode = 246),
			make(code = "FJ", code_iso3116_alpha3 = "FJI", numericCode = 242),
			make(code = "FK", code_iso3116_alpha3 = "FLK", numericCode = 238),
			make(code = "FM", code_iso3116_alpha3 = "FSM", numericCode = 583),
			make(code = "FO", code_iso3116_alpha3 = "FRO", numericCode = 234),
			make(code = "FR", code_iso3116_alpha3 = "FRA", numericCode = 250),
			make(code = "GA", code_iso3116_alpha3 = "GAB", numericCode = 266),
			make(code = "GB", code_iso3116_alpha3 = "GBR", numericCode = 826),
			make(code = "GD", code_iso3116_alpha3 = "GRD", numericCode = 308),
			make(code = "GE", code_iso3116_alpha3 = "GEO", numericCode = 268),
			make(code = "GF", code_iso3116_alpha3 = "GUF", numericCode = 254),
			make(code = "GG", code_iso3116_alpha3 = "GGY", numericCode = 831),
			make(code = "GH", code_iso3116_alpha3 = "GHA", numericCode = 288),
			make(code = "GI", code_iso3116_alpha3 = "GIB", numericCode = 292),
			make(code = "GL", code_iso3116_alpha3 = "GRL", numericCode = 304),
			make(code = "GM", code_iso3116_alpha3 = "GMB", numericCode = 270),
			make(code = "GN", code_iso3116_alpha3 = "GIN", numericCode = 324),
			make(code = "GP", code_iso3116_alpha3 = "GLP", numericCode = 312),
			make(code = "GQ", code_iso3116_alpha3 = "GNQ", numericCode = 226),
			make(code = "GR", code_iso3116_alpha3 = "GRC", numericCode = 300),
			make(code = "GS", code_iso3116_alpha3 = "SGS", numericCode = 239),
			make(code = "GT", code_iso3116_alpha3 = "GTM", numericCode = 320),
			make(code = "GU", code_iso3116_alpha3 = "GUM", numericCode = 316),
			make(code = "GW", code_iso3116_alpha3 = "GNB", numericCode = 624),
			make(code = "GY", code_iso3116_alpha3 = "GUY", numericCode = 328),
			make(code = "HK", code_iso3116_alpha3 = "HKG", numericCode = 344),
			make(code = "HM", code_iso3116_alpha3 = "HMD", numericCode = 334),
			make(code = "HN", code_iso3116_alpha3 = "HND", numericCode = 340),
			make(code = "HR", code_iso3116_alpha3 = "HRV", numericCode = 191),
			make(code = "HT", code_iso3116_alpha3 = "HTI", numericCode = 332),
			make(code = "HU", code_iso3116_alpha3 = "HUN", numericCode = 348),
			make(code = "ID", code_iso3116_alpha3 = "IDN", numericCode = 360),
			make(code = "IE", code_iso3116_alpha3 = "IRL", numericCode = 372),
			make(code = "IL", code_iso3116_alpha3 = "ISR", numericCode = 376),
			make(code = "IM", code_iso3116_alpha3 = "IMN", numericCode = 833),
			make(code = "IN", code_iso3116_alpha3 = "IND", numericCode = 356),
			make(code = "IO", code_iso3116_alpha3 = "IOT", numericCode = 86),
			make(code = "IQ", code_iso3116_alpha3 = "IRQ", numericCode = 368),
			make(code = "IR", code_iso3116_alpha3 = "IRN", numericCode = 364),
			make(code = "IS", code_iso3116_alpha3 = "ISL", numericCode = 352),
			make(code = "IT", code_iso3116_alpha3 = "ITA", numericCode = 380),
			make(code = "JE", code_iso3116_alpha3 = "JEY", numericCode = 832),
			make(code = "JM", code_iso3116_alpha3 = "JAM", numericCode = 388),
			make(code = "JO", code_iso3116_alpha3 = "JOR", numericCode = 400),
			make(code = "JP", code_iso3116_alpha3 = "JPN", numericCode = 392),
			make(code = "KE", code_iso3116_alpha3 = "KEN", numericCode = 404),
			make(code = "KG", code_iso3116_alpha3 = "KGZ", numericCode = 417),
			make(code = "KH", code_iso3116_alpha3 = "KHM", numericCode = 116),
			make(code = "KI", code_iso3116_alpha3 = "KIR", numericCode = 296),
			make(code = "KM", code_iso3116_alpha3 = "COM", numericCode = 174),
			make(code = "KN", code_iso3116_alpha3 = "KNA", numericCode = 659),
			make(code = "KP", code_iso3116_alpha3 = "PRK", numericCode = 408),
			make(code = "KR", code_iso3116_alpha3 = "KOR", numericCode = 410),
			make(code = "KW", code_iso3116_alpha3 = "KWT", numericCode = 414),
			make(code = "KY", code_iso3116_alpha3 = "CYM", numericCode = 136),
			make(code = "KZ", code_iso3116_alpha3 = "KAZ", numericCode = 398),
			make(code = "LA", code_iso3116_alpha3 = "LAO", numericCode = 418),
			make(code = "LB", code_iso3116_alpha3 = "LBN", numericCode = 422),
			make(code = "LC", code_iso3116_alpha3 = "LCA", numericCode = 662),
			make(code = "LI", code_iso3116_alpha3 = "LIE", numericCode = 438),
			make(code = "LK", code_iso3116_alpha3 = "LKA", numericCode = 144),
			make(code = "LR", code_iso3116_alpha3 = "LBR", numericCode = 430),
			make(code = "LS", code_iso3116_alpha3 = "LSO", numericCode = 426),
			make(code = "LT", code_iso3116_alpha3 = "LTU", numericCode = 440),
			make(code = "LU", code_iso3116_alpha3 = "LUX", numericCode = 442),
			make(code = "LV", code_iso3116_alpha3 = "LVA", numericCode = 428),
			make(code = "LY", code_iso3116_alpha3 = "LBY", numericCode = 434),
			make(code = "MA", code_iso3116_alpha3 = "MAR", numericCode = 504),
			make(code = "MC", code_iso3116_alpha3 = "MCO", numericCode = 492),
			make(code = "MD", code_iso3116_alpha3 = "MDA", numericCode = 498),
			make(code = "ME", code_iso3116_alpha3 = "MNE", numericCode = 499),
			make(code = "MF", code_iso3116_alpha3 = "MAF", numericCode = 663),
			make(code = "MG", code_iso3116_alpha3 = "MDG", numericCode = 450),
			make(code = "MH", code_iso3116_alpha3 = "MHL", numericCode = 584),
			make(code = "MK", code_iso3116_alpha3 = "MKD", numericCode = 807),
			make(code = "ML", code_iso3116_alpha3 = "MLI", numericCode = 466),
			make(code = "MM", code_iso3116_alpha3 = "MMR", numericCode = 104),
			make(code = "MN", code_iso3116_alpha3 = "MNG", numericCode = 496),
			make(code = "MO", code_iso3116_alpha3 = "MAC", numericCode = 446),
			make(code = "MP", code_iso3116_alpha3 = "MNP", numericCode = 580),
			make(code = "MQ", code_iso3116_alpha3 = "MTQ", numericCode = 474),
			make(code = "MR", code_iso3116_alpha3 = "MRT", numericCode = 478),
			make(code = "MS", code_iso3116_alpha3 = "MSR", numericCode = 500),
			make(code = "MT", code_iso3116_alpha3 = "MLT", numericCode = 470),
			make(code = "MU", code_iso3116_alpha3 = "MUS", numericCode = 480),
			make(code = "MV", code_iso3116_alpha3 = "MDV", numericCode = 462),
			make(code = "MW", code_iso3116_alpha3 = "MWI", numericCode = 454),
			make(code = "MX", code_iso3116_alpha3 = "MEX", numericCode = 484),
			make(code = "MY", code_iso3116_alpha3 = "MYS", numericCode = 458),
			make(code = "MZ", code_iso3116_alpha3 = "MOZ", numericCode = 508),
			make(code = "NA", code_iso3116_alpha3 = "NAM", numericCode = 516),
			make(code = "NC", code_iso3116_alpha3 = "NCL", numericCode = 540),
			make(code = "NE", code_iso3116_alpha3 = "NER", numericCode = 562),
			make(code = "NF", code_iso3116_alpha3 = "NFK", numericCode = 574),
			make(code = "NG", code_iso3116_alpha3 = "NGA", numericCode = 566),
			make(code = "NI", code_iso3116_alpha3 = "NIC", numericCode = 558),
			make(code = "NL", code_iso3116_alpha3 = "NLD", numericCode = 528),
			make(code = "NO", code_iso3116_alpha3 = "NOR", numericCode = 578),
			make(code = "NP", code_iso3116_alpha3 = "NPL", numericCode = 524),
			make(code = "NR", code_iso3116_alpha3 = "NRU", numericCode = 520),
			make(code = "NU", code_iso3116_alpha3 = "NIU", numericCode = 570),
			make(code = "NZ", code_iso3116_alpha3 = "NZL", numericCode = 554),
			make(code = "OM", code_iso3116_alpha3 = "OMN", numericCode = 512),
			make(code = "PA", code_iso3116_alpha3 = "PAN", numericCode = 591),
			make(code = "PE", code_iso3116_alpha3 = "PER", numericCode = 604),
			make(code = "PF", code_iso3116_alpha3 = "PYF", numericCode = 258),
			make(code = "PG", code_iso3116_alpha3 = "PNG", numericCode = 598),
			make(code = "PH", code_iso3116_alpha3 = "PHL", numericCode = 608),
			make(code = "PK", code_iso3116_alpha3 = "PAK", numericCode = 586),
			make(code = "PL", code_iso3116_alpha3 = "POL", numericCode = 616),
			make(code = "PM", code_iso3116_alpha3 = "SPM", numericCode = 666),
			make(code = "PN", code_iso3116_alpha3 = "PCN", numericCode = 612),
			make(code = "PR", code_iso3116_alpha3 = "PRI", numericCode = 630),
			make(code = "PS", code_iso3116_alpha3 = "PSE", numericCode = 275),
			make(code = "PT", code_iso3116_alpha3 = "PRT", numericCode = 620),
			make(code = "PW", code_iso3116_alpha3 = "PLW", numericCode = 585),
			make(code = "PY", code_iso3116_alpha3 = "PRY", numericCode = 600),
			make(code = "QA", code_iso3116_alpha3 = "QAT", numericCode = 634),
			make(code = "RE", code_iso3116_alpha3 = "REU", numericCode = 638),
			make(code = "RO", code_iso3116_alpha3 = "ROU", numericCode = 642),
			make(code = "RS", code_iso3116_alpha3 = "SRB", numericCode = 688),
			make(code = "RU", code_iso3116_alpha3 = "RUS", numericCode = 643),
			make(code = "RW", code_iso3116_alpha3 = "RWA", numericCode = 646),
			make(code = "SA", code_iso3116_alpha3 = "SAU", numericCode = 682),
			make(code = "SB", code_iso3116_alpha3 = "SLB", numericCode = 90),
			make(code = "SC", code_iso3116_alpha3 = "SYC", numericCode = 690),
			make(code = "SD", code_iso3116_alpha3 = "SDN", numericCode = 729),
			make(code = "SE", code_iso3116_alpha3 = "SWE", numericCode = 752),
			make(code = "SG", code_iso3116_alpha3 = "SGP", numericCode = 702),
			make(code = "SH", code_iso3116_alpha3 = "SHN", numericCode = 654),
			make(code = "SI", code_iso3116_alpha3 = "SVN", numericCode = 705),
			make(code = "SJ", code_iso3116_alpha3 = "SJM", numericCode = 744),
			make(code = "SK", code_iso3116_alpha3 = "SVK", numericCode = 703),
			make(code = "SL", code_iso3116_alpha3 = "SLE", numericCode = 694),
			make(code = "SM", code_iso3116_alpha3 = "SMR", numericCode = 674),
			make(code = "SN", code_iso3116_alpha3 = "SEN", numericCode = 686),
			make(code = "SO", code_iso3116_alpha3 = "SOM", numericCode = 706),
			make(code = "SR", code_iso3116_alpha3 = "SUR", numericCode = 740),
			make(code = "SS", code_iso3116_alpha3 = "SSD", numericCode = 728),
			make(code = "ST", code_iso3116_alpha3 = "STP", numericCode = 678),
			make(code = "SV", code_iso3116_alpha3 = "SLV", numericCode = 222),
			make(code = "SX", code_iso3116_alpha3 = "SXM", numericCode = 534),
			make(code = "SY", code_iso3116_alpha3 = "SYR", numericCode = 760),
			make(code = "SZ", code_iso3116_alpha3 = "SWZ", numericCode = 748),
			make(code = "TC", code_iso3116_alpha3 = "TCA", numericCode = 796),
			make(code = "TD", code_iso3116_alpha3 = "TCD", numericCode = 148),
			make(code = "TF", code_iso3116_alpha3 = "ATF", numericCode = 260),
			make(code = "TG", code_iso3116_alpha3 = "TGO", numericCode = 768),
			make(code = "TH", code_iso3116_alpha3 = "THA", numericCode = 764),
			make(code = "TJ", code_iso3116_alpha3 = "TJK", numericCode = 762),
			make(code = "TK", code_iso3116_alpha3 = "TKL", numericCode = 772),
			make(code = "TL", code_iso3116_alpha3 = "TLS", numericCode = 626),
			make(code = "TM", code_iso3116_alpha3 = "TKM", numericCode = 795),
			make(code = "TN", code_iso3116_alpha3 = "TUN", numericCode = 788),
			make(code = "TO", code_iso3116_alpha3 = "TON", numericCode = 776),
			make(code = "TR", code_iso3116_alpha3 = "TUR", numericCode = 792),
			make(code = "TT", code_iso3116_alpha3 = "TTO", numericCode = 780),
			make(code = "TV", code_iso3116_alpha3 = "TUV", numericCode = 798),
			make(code = "TW", code_iso3116_alpha3 = "TWN", numericCode = 158),
			make(code = "TZ", code_iso3116_alpha3 = "TZA", numericCode = 834),
			make(code = "UA", code_iso3116_alpha3 = "UKR", numericCode = 804),
			make(code = "UG", code_iso3116_alpha3 = "UGA", numericCode = 800),
			make(code = "UM", code_iso3116_alpha3 = "UMI", numericCode = 581),
			make(code = "US", code_iso3116_alpha3 = "USA", numericCode = 840),
			make(code = "UY", code_iso3116_alpha3 = "URY", numericCode = 858),
			make(code = "UZ", code_iso3116_alpha3 = "UZB", numericCode = 860),
			make(code = "VA", code_iso3116_alpha3 = "VAT", numericCode = 336),
			make(code = "VC", code_iso3116_alpha3 = "VCT", numericCode = 670),
			make(code = "VE", code_iso3116_alpha3 = "VEN", numericCode = 862),
			make(code = "VG", code_iso3116_alpha3 = "VGB", numericCode = 92),
			make(code = "VI", code_iso3116_alpha3 = "VIR", numericCode = 850),
			make(code = "VN", code_iso3116_alpha3 = "VNM", numericCode = 704),
			make(code = "VU", code_iso3116_alpha3 = "VUT", numericCode = 548),
			make(code = "WF", code_iso3116_alpha3 = "WLF", numericCode = 876),
			make(code = "WS", code_iso3116_alpha3 = "WSM", numericCode = 882),
			make(code = "YE", code_iso3116_alpha3 = "YEM", numericCode = 887),
			make(code = "YT", code_iso3116_alpha3 = "MYT", numericCode = 175),
			make(code = "ZA", code_iso3116_alpha3 = "ZAF", numericCode = 710),
			make(code = "ZM", code_iso3116_alpha3 = "ZMB", numericCode = 894),
			make(code = "ZW", code_iso3116_alpha3 = "ZWE", numericCode = 716),
		)


		private val allByCode: Map<CountryCode, Country> = all.associateByTo(hashMapOf()) { it.code }


		private fun make(code: String, code_iso3116_alpha3: String, numericCode: Int): Country =
			Country(
				code = CountryCode(code),
				code_iso3116_alpha3 = code_iso3116_alpha3,
				numericCode = numericCode
			)


		public fun forCode(code: CountryCode): Country =
			forCodeOrNull(code) ?: error("Invalid ISO 3166-1 alpha-2 country code: $code")


		public fun forCodeOrNull(code: CountryCode): Country? =
			allByCode[code]


		public fun forCode(code: String): Country =
			forCodeOrNull(CountryCode.parse(code)) ?: error("Invalid ISO 3166-1 alpha-2 country code: $code")


		public fun forCodeOrNull(code: String): Country? =
			CountryCode.parseOrNull(code)?.let(::forCodeOrNull)
	}
}
