/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.clnt.v5_0.extended.leaderelection;

import io.fabric8.kubernetes.clnt.v5_0.KubernetesClient;
import io.fabric8.kubernetes.clnt.v5_0.dsl.Namespaceable;
import io.fabric8.kubernetes.clnt.v5_0.extended.leaderelection.LeaderElectionConfig;
import io.fabric8.kubernetes.clnt.v5_0.extended.leaderelection.resourcelock.LeaderElectionRecord;
import io.fabric8.kubernetes.clnt.v5_0.extended.leaderelection.resourcelock.Lock;
import io.fabric8.kubernetes.clnt.v5_0.extended.leaderelection.resourcelock.LockException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeaderElector<C extends Namespaceable<C> & KubernetesClient> {
    private static final Logger LOGGER = LoggerFactory.getLogger(LeaderElector.class);
    protected static final Double JITTER_FACTOR = 1.2;
    private C kubernetesClient;
    private LeaderElectionConfig leaderElectionConfig;
    private final AtomicReference<LeaderElectionRecord> observedRecord;
    private final AtomicReference<LocalDateTime> observedTime;
    private final AtomicReference<String> reportedLeader;

    public LeaderElector(C kubernetesClient, LeaderElectionConfig leaderElectionConfig) {
        this.kubernetesClient = kubernetesClient;
        this.leaderElectionConfig = leaderElectionConfig;
        this.observedRecord = new AtomicReference();
        this.observedTime = new AtomicReference();
        this.reportedLeader = new AtomicReference();
    }

    public void run() {
        LOGGER.debug("Leader election started");
        if (!this.acquire()) {
            return;
        }
        this.leaderElectionConfig.getLeaderCallbacks().onStartLeading();
        this.renewWithTimeout();
        this.leaderElectionConfig.getLeaderCallbacks().onStopLeading();
    }

    private boolean acquire() {
        String lockDescription = this.leaderElectionConfig.getLock().describe();
        LOGGER.debug("Attempting to acquire leader lease '{}'...", (Object)lockDescription);
        AtomicBoolean succeeded = new AtomicBoolean(false);
        return LeaderElector.loop(countDownLatch -> {
            try {
                if (!succeeded.get()) {
                    succeeded.set(this.tryAcquireOrRenew());
                    this.reportTransitionIfLeaderChanged();
                }
                if (succeeded.get()) {
                    LOGGER.debug("Successfully Acquired leader lease '{}'", (Object)lockDescription);
                    countDownLatch.countDown();
                } else {
                    LOGGER.debug("Failed to acquire lease '{}' retrying...", (Object)lockDescription);
                }
            }
            catch (Exception exception) {
                LOGGER.error("Exception occurred while acquiring lock '{}'", (Object)lockDescription, (Object)exception);
            }
        }, LeaderElector.jitter(this.leaderElectionConfig.getRetryPeriod(), JITTER_FACTOR).toMillis());
    }

    private void renewWithTimeout() {
        String lockDescription = this.leaderElectionConfig.getLock().describe();
        LOGGER.debug("Attempting to renew leader lease '{}'...", (Object)lockDescription);
        LeaderElector.loop(abortLatch -> {
            ScheduledExecutorService timeoutExecutorService = Executors.newSingleThreadScheduledExecutor();
            CountDownLatch renewSignal = new CountDownLatch(1);
            try {
                timeoutExecutorService.submit(() -> this.renew((CountDownLatch)abortLatch, renewSignal));
                if (!renewSignal.await(this.leaderElectionConfig.getRenewDeadline().toMillis(), TimeUnit.MILLISECONDS)) {
                    LOGGER.debug("Renew deadline reached after {} seconds while renewing lock {}", (Object)this.leaderElectionConfig.getRenewDeadline().get(ChronoUnit.SECONDS), (Object)lockDescription);
                    abortLatch.countDown();
                }
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            finally {
                timeoutExecutorService.shutdown();
            }
        }, this.leaderElectionConfig.getRetryPeriod().toMillis());
    }

    private void renew(CountDownLatch abortLatch, CountDownLatch renewSignal) {
        try {
            boolean success = this.tryAcquireOrRenew();
            this.reportTransitionIfLeaderChanged();
            if (!success) {
                abortLatch.countDown();
            }
        }
        catch (LockException exception) {
            LOGGER.debug("Exception occurred while renewing lock: {}", (Object)exception.getMessage(), (Object)exception);
        }
        renewSignal.countDown();
    }

    private boolean tryAcquireOrRenew() throws LockException {
        Lock lock = this.leaderElectionConfig.getLock();
        ZonedDateTime now = LeaderElector.now();
        LeaderElectionRecord oldLeaderElectionRecord = lock.get(this.kubernetesClient);
        if (oldLeaderElectionRecord == null) {
            LeaderElectionRecord newLeaderElectionRecord = new LeaderElectionRecord(lock.identity(), this.leaderElectionConfig.getLeaseDuration(), now, now, 0);
            lock.create(this.kubernetesClient, newLeaderElectionRecord);
            this.updateObserved(newLeaderElectionRecord);
            return true;
        }
        this.updateObserved(oldLeaderElectionRecord);
        boolean isLeader = this.isLeader(oldLeaderElectionRecord);
        if (!isLeader && !this.canBecomeLeader(oldLeaderElectionRecord)) {
            LOGGER.debug("Lock is held by {} and has not yet expired", (Object)oldLeaderElectionRecord.getHolderIdentity());
            return false;
        }
        LeaderElectionRecord newLeaderElectionRecord = new LeaderElectionRecord(lock.identity(), this.leaderElectionConfig.getLeaseDuration(), isLeader ? oldLeaderElectionRecord.getAcquireTime() : now, now, isLeader ? oldLeaderElectionRecord.getLeaderTransitions() + 1 : 0);
        newLeaderElectionRecord.setVersion(oldLeaderElectionRecord.getVersion());
        this.leaderElectionConfig.getLock().update(this.kubernetesClient, newLeaderElectionRecord);
        this.updateObserved(newLeaderElectionRecord);
        return true;
    }

    private void updateObserved(LeaderElectionRecord leaderElectionRecord) {
        if (!Objects.equals(leaderElectionRecord, this.observedRecord.get())) {
            this.observedRecord.set(leaderElectionRecord);
            this.observedTime.set(LocalDateTime.now());
        }
    }

    private void reportTransitionIfLeaderChanged() {
        String currentLeader = this.reportedLeader.get();
        String newLeader = this.observedRecord.get().getHolderIdentity();
        if (!Objects.equals(newLeader, currentLeader)) {
            LOGGER.debug("Leader changed from {} to {}", (Object)currentLeader, (Object)newLeader);
            this.reportedLeader.set(newLeader);
            this.leaderElectionConfig.getLeaderCallbacks().onNewLeader(newLeader);
        }
    }

    protected final boolean isLeader(LeaderElectionRecord leaderElectionRecord) {
        return Objects.equals(this.leaderElectionConfig.getLock().identity(), leaderElectionRecord.getHolderIdentity());
    }

    protected final boolean canBecomeLeader(LeaderElectionRecord leaderElectionRecord) {
        return !leaderElectionRecord.getRenewTime().plus(this.leaderElectionConfig.getLeaseDuration()).isAfter(LeaderElector.now());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static boolean loop(Consumer<CountDownLatch> consumer, long periodInMillis) {
        ScheduledExecutorService singleThreadExecutorService = Executors.newSingleThreadScheduledExecutor();
        CountDownLatch countDownLatch = new CountDownLatch(1);
        ScheduledFuture<?> future = singleThreadExecutorService.scheduleAtFixedRate(() -> consumer.accept(countDownLatch), 0L, periodInMillis, TimeUnit.MILLISECONDS);
        try {
            countDownLatch.await();
            boolean bl = true;
            return bl;
        }
        catch (InterruptedException e) {
            LOGGER.debug("Loop thread interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
            boolean bl = false;
            return bl;
        }
        finally {
            future.cancel(true);
            singleThreadExecutorService.shutdownNow();
        }
    }

    protected static ZonedDateTime now() {
        return ZonedDateTime.now(ZoneOffset.UTC);
    }

    protected static Duration jitter(Duration duration, double maxFactor) {
        maxFactor = maxFactor > 0.0 ? maxFactor : 1.0;
        return duration.plusMillis(Double.valueOf((double)duration.toMillis() * Math.random() * maxFactor).longValue());
    }
}

