/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.dsl.internal;

import io.fabric8.kubernetes.api.model.HasMetadata;
import io.fabric8.kubernetes.api.model.KubernetesResource;
import io.fabric8.kubernetes.api.model.KubernetesResourceList;
import io.fabric8.kubernetes.api.model.ListOptions;
import io.fabric8.kubernetes.api.model.Status;
import io.fabric8.kubernetes.api.model.WatchEvent;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.WatcherException;
import io.fabric8.kubernetes.client.dsl.base.BaseOperation;
import io.fabric8.kubernetes.client.dsl.base.OperationSupport;
import io.fabric8.kubernetes.client.dsl.internal.AbstractWatchManager;
import io.fabric8.kubernetes.client.dsl.internal.NamedRunnable;
import io.fabric8.kubernetes.client.dsl.internal.WatchHTTPManager;
import io.fabric8.kubernetes.client.utils.HttpClientUtils;
import io.fabric8.kubernetes.client.utils.Utils;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import okio.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WatchConnectionManager<T extends HasMetadata, L extends KubernetesResourceList<T>>
extends AbstractWatchManager<T> {
    private static final Logger logger = LoggerFactory.getLogger(WatchConnectionManager.class);
    private final BaseOperation<T, L, ?> baseOperation;
    private final AtomicReference<WebSocket> webSocketRef = new AtomicReference();
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final AtomicBoolean reconnectPending = new AtomicBoolean(false);
    private final ArrayBlockingQueue<Object> queue = new ArrayBlockingQueue(1);
    private final URL requestUrl;

    public WatchConnectionManager(OkHttpClient client, BaseOperation<T, L, ?> baseOperation, ListOptions listOptions, Watcher<T> watcher, int reconnectInterval, int reconnectLimit, long websocketTimeout, int maxIntervalExponent) throws MalformedURLException {
        super(watcher, listOptions, reconnectLimit, reconnectInterval, maxIntervalExponent, client.newBuilder().readTimeout(websocketTimeout, TimeUnit.MILLISECONDS).build());
        this.baseOperation = baseOperation;
        this.requestUrl = baseOperation.getNamespacedUrl();
        this.runWatch();
    }

    public WatchConnectionManager(OkHttpClient client, BaseOperation<T, L, ?> baseOperation, ListOptions listOptions, Watcher<T> watcher, int reconnectInterval, int reconnectLimit, long websocketTimeout) throws MalformedURLException {
        this(client, baseOperation, listOptions, watcher, reconnectInterval, reconnectLimit, websocketTimeout, 5);
    }

    private void runWatch() {
        logger.debug("Connecting websocket to {}...", (Object)this.requestUrl);
        HttpUrl.Builder httpUrlBuilder = HttpUrl.get((URL)this.requestUrl).newBuilder();
        String labelQueryParam = this.baseOperation.getLabelQueryParam();
        if (Utils.isNotNullOrEmpty(labelQueryParam)) {
            httpUrlBuilder.addQueryParameter("labelSelector", labelQueryParam);
        }
        String fieldQueryString = this.baseOperation.getFieldQueryParam();
        String name = this.baseOperation.getName();
        if (name != null && name.length() > 0) {
            if (this.baseOperation.isApiGroup()) {
                httpUrlBuilder.addPathSegment(name);
            } else {
                if (fieldQueryString.length() > 0) {
                    fieldQueryString = fieldQueryString + ",";
                }
                fieldQueryString = fieldQueryString + "metadata.name=" + name;
            }
        }
        if (Utils.isNotNullOrEmpty(fieldQueryString)) {
            if (this.baseOperation.isApiGroup()) {
                logger.warn("Ignoring field selector " + fieldQueryString + " on watch URI " + this.requestUrl + " as fieldSelector is not yet supported on API Groups APIs");
            } else {
                httpUrlBuilder.addQueryParameter("fieldSelector", fieldQueryString);
            }
        }
        this.listOptions.setResourceVersion((String)this.resourceVersion.get());
        HttpClientUtils.appendListOptionParams(httpUrlBuilder, this.listOptions);
        String origin = this.requestUrl.getProtocol() + "://" + this.requestUrl.getHost();
        if (this.requestUrl.getPort() != -1) {
            origin = origin + ":" + this.requestUrl.getPort();
        }
        Request request = new Request.Builder().get().url(httpUrlBuilder.build()).addHeader("Origin", origin).build();
        this.clonedClient.newWebSocket(request, new WebSocketListener(){

            public void onOpen(WebSocket webSocket, Response response) {
                if (response != null && response.body() != null) {
                    response.body().close();
                }
                logger.debug("WebSocket successfully opened");
                WatchConnectionManager.this.webSocketRef.set(webSocket);
                WatchConnectionManager.this.currentReconnectAttempt.set(0);
                WatchConnectionManager.this.started.set(true);
                WatchConnectionManager.this.queue.clear();
                WatchConnectionManager.this.queue.add(true);
            }

            public void onFailure(WebSocket webSocket, Throwable t, Response response) {
                if (WatchConnectionManager.this.forceClosed.get()) {
                    logger.debug("Ignoring onFailure for already closed/closing websocket", t);
                    if (response != null && response.body() != null) {
                        response.body().close();
                    }
                    return;
                }
                if (response != null && (response.code() == 200 || response.code() == 503)) {
                    WatchConnectionManager.this.queue.clear();
                    WatchConnectionManager.this.queue.offer(new KubernetesClientException("Received " + response.code() + " on websocket", response.code(), null));
                    response.body().close();
                    return;
                }
                if (response != null) {
                    Status status = OperationSupport.createStatus(response);
                    if (response.body() != null) {
                        response.body().close();
                    }
                    logger.warn("Exec Failure: HTTP {}, Status: {} - {}", new Object[]{response.code(), status.getCode(), status.getMessage(), t});
                    if (!WatchConnectionManager.this.started.get()) {
                        WatchConnectionManager.this.queue.clear();
                        WatchConnectionManager.this.queue.offer(new KubernetesClientException(status));
                    }
                } else {
                    logger.warn("Exec Failure", t);
                    if (!WatchConnectionManager.this.started.get()) {
                        WatchConnectionManager.this.queue.clear();
                        WatchConnectionManager.this.queue.offer(new KubernetesClientException("Failed to start websocket", t));
                    }
                }
                if (WatchConnectionManager.this.cannotReconnect()) {
                    WatchConnectionManager.this.closeEvent(new WatcherException("Connection failure", t));
                    return;
                }
                WatchConnectionManager.this.scheduleReconnect();
            }

            public void onMessage(WebSocket webSocket, ByteString bytes) {
                this.onMessage(webSocket, bytes.utf8());
            }

            public void onMessage(WebSocket webSocket, String message) {
                try {
                    WatchEvent event = WatchHTTPManager.readWatchEvent(message);
                    KubernetesResource object = event.getObject();
                    if (object instanceof HasMetadata) {
                        HasMetadata obj = (HasMetadata)object;
                        WatchConnectionManager.this.resourceVersion.set(obj.getMetadata().getResourceVersion());
                        Watcher.Action action = Watcher.Action.valueOf(event.getType());
                        WatchConnectionManager.this.watcher.eventReceived(action, obj);
                    } else if (object instanceof KubernetesResourceList) {
                        KubernetesResourceList list = (KubernetesResourceList)object;
                        WatchConnectionManager.this.resourceVersion.set(list.getMetadata().getResourceVersion());
                        Watcher.Action action = Watcher.Action.valueOf(event.getType());
                        List items = list.getItems();
                        if (items != null) {
                            for (HasMetadata item : items) {
                                WatchConnectionManager.this.watcher.eventReceived(action, item);
                            }
                        }
                    } else if (object instanceof Status) {
                        Status status = (Status)object;
                        if (status.getCode() == 410) {
                            WatchConnectionManager.this.webSocketRef.set(null);
                            WatchConnectionManager.this.closeEvent(new WatcherException(status.getMessage(), new KubernetesClientException(status)));
                            WatchConnectionManager.this.close();
                            return;
                        }
                        WatchConnectionManager.this.watcher.eventReceived(Watcher.Action.ERROR, null);
                        logger.error("Error received: {}", (Object)status);
                    } else {
                        logger.error("Unknown message received: {}", (Object)message);
                    }
                }
                catch (ClassCastException e) {
                    logger.error("Received wrong type of object for watch", (Throwable)e);
                }
                catch (IllegalArgumentException e) {
                    logger.error("Invalid event type", (Throwable)e);
                }
            }

            public void onClosing(WebSocket webSocket, int code, String reason) {
                webSocket.close(code, reason);
            }

            public void onClosed(WebSocket webSocket, int code, String reason) {
                logger.debug("WebSocket close received. code: {}, reason: {}", (Object)code, (Object)reason);
                if (WatchConnectionManager.this.forceClosed.get()) {
                    logger.debug("Ignoring onClose for already closed/closing websocket");
                    return;
                }
                if (WatchConnectionManager.this.cannotReconnect()) {
                    WatchConnectionManager.this.closeEvent(new WatcherException("Connection unexpectedly closed"));
                    return;
                }
                WatchConnectionManager.this.scheduleReconnect();
            }
        });
    }

    private void scheduleReconnect() {
        logger.debug("Submitting reconnect task to the executor");
        this.submit(new NamedRunnable("scheduleReconnect"){

            @Override
            public void execute() {
                if (!WatchConnectionManager.this.reconnectPending.compareAndSet(false, true)) {
                    logger.debug("Reconnect already scheduled");
                    return;
                }
                WatchConnectionManager.this.webSocketRef.set(null);
                try {
                    logger.debug("Scheduling reconnect task");
                    WatchConnectionManager.this.schedule(new NamedRunnable("reconnectAttempt"){

                        @Override
                        public void execute() {
                            try {
                                WatchConnectionManager.this.runWatch();
                                WatchConnectionManager.this.reconnectPending.set(false);
                            }
                            catch (Exception e) {
                                logger.error("Exception in reconnect", (Throwable)e);
                                WatchConnectionManager.this.webSocketRef.set(null);
                                WatchConnectionManager.this.closeEvent(new WatcherException("Unhandled exception in reconnect attempt", e));
                                WatchConnectionManager.this.close();
                            }
                        }
                    }, WatchConnectionManager.this.nextReconnectInterval(), TimeUnit.MILLISECONDS);
                }
                catch (RejectedExecutionException e) {
                    WatchConnectionManager.this.reconnectPending.set(false);
                }
            }
        });
    }

    public void waitUntilReady() {
        Utils.waitUntilReady(this.queue, 10L, TimeUnit.SECONDS);
    }

    @Override
    public void close() {
        logger.debug("Force closing the watch {}", (Object)this);
        this.closeEvent();
        WatchConnectionManager.closeWebSocket(this.webSocketRef.getAndSet(null));
        this.closeExecutorService();
    }
}

