
/**
 * Copyright (C) 2015 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

                        

package io.fabric8.kubernetes.client.handlers;

import java.util.function.Predicate;

import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.ResourceHandler;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.internal.ConfigMapOperationsImpl;

import io.fabric8.kubernetes.client.dsl.base.OperationContext;
import okhttp3.OkHttpClient;

import io.fabric8.kubernetes.api.model.ConfigMap;
import io.fabric8.kubernetes.api.model.ConfigMapBuilder;

import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

public class ConfigMapHandler implements ResourceHandler<ConfigMap, ConfigMapBuilder> {

  @Override
  public String getKind() {
    return ConfigMap.class.getSimpleName();
  }

  @Override
  public String getApiVersion() {
        return "v1";
      }

  @Override
  public ConfigMap create(OkHttpClient client, Config config, String namespace, ConfigMap item) {
    return new ConfigMapOperationsImpl(client, config).withItem(item).inNamespace(namespace).create();
  }

  @Override
  public ConfigMap replace(OkHttpClient client, Config config, String namespace, ConfigMap item) {
    return new ConfigMapOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).replace(item);
  }

  @Override
  public ConfigMap reload(OkHttpClient client, Config config, String namespace, ConfigMap item) {
    return new ConfigMapOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).fromServer().get();
  }

  @Override
  public ConfigMapBuilder edit(ConfigMap item) {
    return new ConfigMapBuilder(item);
  }

  @Override
  public Boolean delete(OkHttpClient client, Config config, String namespace, Boolean cascading, ConfigMap item) {
    return new ConfigMapOperationsImpl(client, config, namespace).withItem(item).cascading(cascading).delete();
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, ConfigMap item, Watcher<ConfigMap> watcher) {
    return new ConfigMapOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(watcher);
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, ConfigMap item, String resourceVersion, Watcher<ConfigMap> watcher) {
    return new ConfigMapOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(resourceVersion, watcher);
  }

  @Override
  public ConfigMap waitUntilReady(OkHttpClient client, Config config, String namespace, ConfigMap item, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new ConfigMapOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilReady(amount, timeUnit);
  }

  @Override
  public ConfigMap waitUntilCondition(OkHttpClient client, Config config, String namespace, ConfigMap item, Predicate<ConfigMap> condition, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new ConfigMapOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilCondition(condition, amount, timeUnit);
  }
}
