
/**
 * Copyright (C) 2015 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

                        

package io.fabric8.kubernetes.client.handlers;

import java.util.function.Predicate;

import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.ResourceHandler;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.internal.IngressOperationsImpl;

import io.fabric8.kubernetes.client.dsl.base.OperationContext;
import okhttp3.OkHttpClient;

import io.fabric8.kubernetes.api.model.extensions.Ingress;
import io.fabric8.kubernetes.api.model.extensions.IngressBuilder;

import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

public class IngressHandler implements ResourceHandler<Ingress, IngressBuilder> {

  @Override
  public String getKind() {
    return Ingress.class.getSimpleName();
  }

  @Override
  public String getApiVersion() {
        return "extensions/v1beta1";
      }

  @Override
  public Ingress create(OkHttpClient client, Config config, String namespace, Ingress item) {
    return new IngressOperationsImpl(client, config).withItem(item).inNamespace(namespace).create();
  }

  @Override
  public Ingress replace(OkHttpClient client, Config config, String namespace, Ingress item) {
    return new IngressOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).replace(item);
  }

  @Override
  public Ingress reload(OkHttpClient client, Config config, String namespace, Ingress item) {
    return new IngressOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).fromServer().get();
  }

  @Override
  public IngressBuilder edit(Ingress item) {
    return new IngressBuilder(item);
  }

  @Override
  public Boolean delete(OkHttpClient client, Config config, String namespace, Boolean cascading, Ingress item) {
    return new IngressOperationsImpl(client, config, namespace).withItem(item).cascading(cascading).delete();
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Ingress item, Watcher<Ingress> watcher) {
    return new IngressOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(watcher);
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Ingress item, String resourceVersion, Watcher<Ingress> watcher) {
    return new IngressOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(resourceVersion, watcher);
  }

  @Override
  public Ingress waitUntilReady(OkHttpClient client, Config config, String namespace, Ingress item, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new IngressOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilReady(amount, timeUnit);
  }

  @Override
  public Ingress waitUntilCondition(OkHttpClient client, Config config, String namespace, Ingress item, Predicate<Ingress> condition, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new IngressOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilCondition(condition, amount, timeUnit);
  }
}
