/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.maven.docker.service;

import com.google.common.collect.ImmutableMap;
import io.fabric8.maven.docker.access.BuildOptions;
import io.fabric8.maven.docker.access.DockerAccess;
import io.fabric8.maven.docker.access.DockerAccessException;
import io.fabric8.maven.docker.config.AssemblyConfiguration;
import io.fabric8.maven.docker.config.BuildImageConfiguration;
import io.fabric8.maven.docker.config.CleanupMode;
import io.fabric8.maven.docker.config.ImageConfiguration;
import io.fabric8.maven.docker.service.ArchiveService;
import io.fabric8.maven.docker.service.ImagePullManager;
import io.fabric8.maven.docker.service.QueryService;
import io.fabric8.maven.docker.service.RegistryService;
import io.fabric8.maven.docker.util.DockerFileUtil;
import io.fabric8.maven.docker.util.EnvUtil;
import io.fabric8.maven.docker.util.ImageName;
import io.fabric8.maven.docker.util.Logger;
import io.fabric8.maven.docker.util.MojoParameters;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.apache.maven.plugin.MojoExecutionException;

public class BuildService {
    private final DockerAccess docker;
    private final QueryService queryService;
    private final ArchiveService archiveService;
    private final RegistryService registryService;
    private final Logger log;

    BuildService(DockerAccess docker, QueryService queryService, RegistryService registryService, ArchiveService archiveService, Logger log) {
        this.docker = docker;
        this.queryService = queryService;
        this.registryService = registryService;
        this.archiveService = archiveService;
        this.log = log;
    }

    public void buildImage(ImageConfiguration imageConfig, ImagePullManager imagePullManager, BuildContext buildContext) throws DockerAccessException, MojoExecutionException {
        if (imagePullManager != null) {
            this.autoPullBaseImage(imageConfig, imagePullManager, buildContext);
        }
        this.buildImage(imageConfig, buildContext.getMojoParameters(), this.checkForNocache(imageConfig), this.addBuildArgs(buildContext));
    }

    public void tagImage(String imageName, ImageConfiguration imageConfig) throws DockerAccessException {
        List<String> tags = imageConfig.getBuildConfiguration().getTags();
        if (tags.size() > 0) {
            this.log.info("%s: Tag with %s", imageConfig.getDescription(), EnvUtil.stringJoin(tags, ","));
            for (String tag : tags) {
                if (tag == null) continue;
                this.docker.tag(imageName, new ImageName(imageName, tag).getFullName(), true);
            }
            this.log.debug("Tagging image successful!", new Object[0]);
        }
    }

    protected void buildImage(ImageConfiguration imageConfig, MojoParameters params, boolean noCache, Map<String, String> buildArgs) throws DockerAccessException, MojoExecutionException {
        String imageName = imageConfig.getName();
        ImageName.validate(imageName);
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        String oldImageId = null;
        CleanupMode cleanupMode = buildConfig.cleanupMode();
        if (cleanupMode.isRemove()) {
            oldImageId = this.queryService.getImageId(imageName);
        }
        long time = System.currentTimeMillis();
        if (buildConfig.getDockerArchive() != null) {
            this.docker.loadImage(imageName, buildConfig.getAbsoluteDockerTarPath(params));
            this.log.info("%s: Loaded tarball in %s", buildConfig.getDockerArchive(), EnvUtil.formatDurationTill(time));
            return;
        }
        File dockerArchive = this.archiveService.createArchive(imageName, buildConfig, params, this.log);
        this.log.info("%s: Created %s in %s", imageConfig.getDescription(), dockerArchive.getName(), EnvUtil.formatDurationTill(time));
        Map<String, String> mergedBuildMap = this.prepareBuildArgs(buildArgs, buildConfig);
        BuildOptions opts = new BuildOptions(buildConfig.getBuildOptions()).dockerfile(this.getDockerfileName(buildConfig)).forceRemove(cleanupMode.isRemove()).noCache(noCache).buildArgs(mergedBuildMap);
        String newImageId = this.doBuildImage(imageName, dockerArchive, opts);
        this.log.info("%s: Built image %s", imageConfig.getDescription(), newImageId);
        if (oldImageId != null && !oldImageId.equals(newImageId)) {
            try {
                this.docker.removeImage(oldImageId, true);
                this.log.info("%s: Removed old image %s", imageConfig.getDescription(), oldImageId);
            }
            catch (DockerAccessException exp) {
                if (cleanupMode == CleanupMode.TRY_TO_REMOVE) {
                    this.log.warn("%s: %s (old image)%s", imageConfig.getDescription(), exp.getMessage(), exp.getCause() != null ? " [" + exp.getCause().getMessage() + "]" : "");
                }
                throw exp;
            }
        }
    }

    private Map<String, String> prepareBuildArgs(Map<String, String> buildArgs, BuildImageConfiguration buildConfig) {
        ImmutableMap.Builder builder = ImmutableMap.builder().putAll(buildArgs);
        if (buildConfig.getArgs() != null) {
            builder.putAll(buildConfig.getArgs());
        }
        return builder.build();
    }

    private String getDockerfileName(BuildImageConfiguration buildConfig) {
        if (buildConfig.isDockerFileMode()) {
            return buildConfig.getDockerFile().getName();
        }
        return null;
    }

    private String doBuildImage(String imageName, File dockerArchive, BuildOptions options) throws DockerAccessException, MojoExecutionException {
        this.docker.buildImage(imageName, dockerArchive, options);
        return this.queryService.getImageId(imageName);
    }

    private Map<String, String> addBuildArgs(BuildContext buildContext) {
        Map<String, String> buildArgsFromProject = this.addBuildArgsFromProperties(buildContext.getMojoParameters().getProject().getProperties());
        Map<String, String> buildArgsFromSystem = this.addBuildArgsFromProperties(System.getProperties());
        return ImmutableMap.builder().putAll(buildContext.getBuildArgs() != null ? buildContext.getBuildArgs() : Collections.emptyMap()).putAll(buildArgsFromProject).putAll(buildArgsFromSystem).build();
    }

    private Map<String, String> addBuildArgsFromProperties(Properties properties) {
        String argPrefix = "docker.buildArg.";
        HashMap<String, String> buildArgs = new HashMap<String, String>();
        for (Object keyObj : properties.keySet()) {
            String key = (String)keyObj;
            if (!key.startsWith(argPrefix)) continue;
            String argKey = key.replaceFirst(argPrefix, "");
            String value = properties.getProperty(key);
            if (this.isEmpty(value)) continue;
            buildArgs.put(argKey, value);
        }
        this.log.debug("Build args set %s", buildArgs);
        return buildArgs;
    }

    private void autoPullBaseImage(ImageConfiguration imageConfig, ImagePullManager imagePullManager, BuildContext buildContext) throws DockerAccessException, MojoExecutionException {
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        if (buildConfig.getDockerArchive() != null) {
            return;
        }
        String fromImage = buildConfig.isDockerFileMode() ? this.extractBaseFromDockerfile(buildConfig, buildContext) : this.extractBaseFromConfiguration(buildConfig);
        if (fromImage != null && !"scratch".equals(fromImage)) {
            this.registryService.pullImageWithPolicy(fromImage, imagePullManager, buildContext.getRegistryConfig(), this.queryService.hasImage(fromImage));
        }
    }

    private String extractBaseFromConfiguration(BuildImageConfiguration buildConfig) {
        AssemblyConfiguration assemblyConfig;
        String fromImage = buildConfig.getFrom();
        if (fromImage == null && (assemblyConfig = buildConfig.getAssemblyConfiguration()) == null) {
            fromImage = "busybox:latest";
        }
        return fromImage;
    }

    private String extractBaseFromDockerfile(BuildImageConfiguration buildConfig, BuildContext buildContext) {
        String fromImage;
        try {
            File fullDockerFilePath = buildConfig.getAbsoluteDockerFilePath(buildContext.getMojoParameters());
            fromImage = DockerFileUtil.extractBaseImage(fullDockerFilePath, DockerFileUtil.createInterpolator(buildContext.getMojoParameters(), buildConfig.getFilter()));
        }
        catch (IOException e) {
            fromImage = null;
        }
        return fromImage;
    }

    private boolean checkForNocache(ImageConfiguration imageConfig) {
        String nocache = System.getProperty("docker.nocache");
        if (nocache != null) {
            return nocache.length() == 0 || Boolean.valueOf(nocache) != false;
        }
        BuildImageConfiguration buildConfig = imageConfig.getBuildConfiguration();
        return buildConfig.nocache();
    }

    private boolean isEmpty(String str) {
        return str == null || str.isEmpty();
    }

    public static class BuildContext
    implements Serializable {
        private MojoParameters mojoParameters;
        private Map<String, String> buildArgs;
        private RegistryService.RegistryConfig registryConfig;

        public MojoParameters getMojoParameters() {
            return this.mojoParameters;
        }

        public Map<String, String> getBuildArgs() {
            return this.buildArgs;
        }

        public RegistryService.RegistryConfig getRegistryConfig() {
            return this.registryConfig;
        }

        public static class Builder {
            private BuildContext context = new BuildContext();

            public Builder() {
                this.context = new BuildContext();
            }

            public Builder(BuildContext context) {
                this.context = context;
            }

            public Builder mojoParameters(MojoParameters mojoParameters) {
                this.context.mojoParameters = mojoParameters;
                return this;
            }

            public Builder buildArgs(Map<String, String> buildArgs) {
                this.context.buildArgs = buildArgs;
                return this;
            }

            public Builder registryConfig(RegistryService.RegistryConfig registryConfig) {
                this.context.registryConfig = registryConfig;
                return this;
            }

            public BuildContext build() {
                return this.context;
            }
        }
    }
}

