/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.maven.docker.service;

import io.fabric8.maven.docker.access.ContainerCreateConfig;
import io.fabric8.maven.docker.access.ContainerHostConfig;
import io.fabric8.maven.docker.access.ContainerNetworkingConfig;
import io.fabric8.maven.docker.access.DockerAccess;
import io.fabric8.maven.docker.access.DockerAccessException;
import io.fabric8.maven.docker.access.NetworkCreateConfig;
import io.fabric8.maven.docker.access.PortMapping;
import io.fabric8.maven.docker.config.Arguments;
import io.fabric8.maven.docker.config.ImageConfiguration;
import io.fabric8.maven.docker.config.NetworkConfig;
import io.fabric8.maven.docker.config.RestartPolicy;
import io.fabric8.maven.docker.config.RunImageConfiguration;
import io.fabric8.maven.docker.config.VolumeConfiguration;
import io.fabric8.maven.docker.log.LogOutputSpecFactory;
import io.fabric8.maven.docker.model.Container;
import io.fabric8.maven.docker.model.Network;
import io.fabric8.maven.docker.service.ContainerTracker;
import io.fabric8.maven.docker.service.QueryService;
import io.fabric8.maven.docker.util.EnvUtil;
import io.fabric8.maven.docker.util.Logger;
import io.fabric8.maven.docker.util.PomLabel;
import io.fabric8.maven.docker.util.StartOrderResolver;
import io.fabric8.maven.docker.util.WaitUtil;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

public class RunService {
    private Logger log;
    private final ContainerTracker tracker;
    private DockerAccess docker;
    private QueryService queryService;
    private final LogOutputSpecFactory logConfig;

    public RunService(DockerAccess docker, QueryService queryService, ContainerTracker tracker, LogOutputSpecFactory logConfig, Logger log) {
        this.docker = docker;
        this.queryService = queryService;
        this.tracker = tracker;
        this.log = log;
        this.logConfig = logConfig;
    }

    public String execInContainer(String containerId, String command, ImageConfiguration imageConfiguration) throws DockerAccessException {
        Arguments arguments = new Arguments();
        arguments.setExec(Arrays.asList(EnvUtil.splitOnSpaceWithEscape(command)));
        String execContainerId = this.docker.createExecContainer(containerId, arguments);
        this.docker.startExecContainer(execContainerId, this.logConfig.createSpec(containerId, imageConfiguration));
        return execContainerId;
    }

    public String createAndStartContainer(ImageConfiguration imageConfig, PortMapping portMapping, PomLabel pomLabel, Properties mavenProps) throws DockerAccessException {
        RunImageConfiguration runConfig = imageConfig.getRunConfiguration();
        String imageName = imageConfig.getName();
        String containerName = this.calculateContainerName(imageConfig.getAlias(), runConfig.getNamingStrategy());
        ContainerCreateConfig config = this.createContainerConfig(imageName, runConfig, portMapping, pomLabel, mavenProps);
        String id = this.docker.createContainer(config, containerName);
        this.startContainer(imageConfig, id, pomLabel);
        if (portMapping.needsPropertiesUpdate()) {
            this.updateMappedPortsAndAddresses(id, portMapping);
        }
        return id;
    }

    public void stopContainer(String containerId, ImageConfiguration imageConfig, boolean keepContainer, boolean removeVolumes) throws DockerAccessException {
        ContainerTracker.ContainerShutdownDescriptor descriptor = new ContainerTracker.ContainerShutdownDescriptor(imageConfig, containerId);
        this.shutdown(descriptor, keepContainer, removeVolumes);
    }

    public void stopPreviouslyStartedContainer(String containerId, boolean keepContainer, boolean removeVolumes) throws DockerAccessException {
        ContainerTracker.ContainerShutdownDescriptor descriptor = this.tracker.removeContainer(containerId);
        if (descriptor != null) {
            this.shutdown(descriptor, keepContainer, removeVolumes);
        }
    }

    public void stopStartedContainers(boolean keepContainer, boolean removeVolumes, boolean removeCustomNetworks, PomLabel pomLabel) throws DockerAccessException {
        HashSet<Network> networksToRemove = new HashSet<Network>();
        for (ContainerTracker.ContainerShutdownDescriptor descriptor : this.tracker.removeShutdownDescriptors(pomLabel)) {
            this.collectCustomNetworks(networksToRemove, descriptor, removeCustomNetworks);
            this.shutdown(descriptor, keepContainer, removeVolumes);
        }
        this.removeCustomNetworks(networksToRemove);
    }

    private void collectCustomNetworks(Set<Network> networksToRemove, ContainerTracker.ContainerShutdownDescriptor descriptor, boolean removeCustomNetworks) throws DockerAccessException {
        NetworkConfig config = descriptor.getImageConfiguration().getRunConfiguration().getNetworkingConfig();
        if (removeCustomNetworks && config.isCustomNetwork()) {
            networksToRemove.add(this.queryService.getNetworkByName(config.getCustomNetwork()));
        }
    }

    public String lookupContainer(String lookup) {
        return this.tracker.lookupContainer(lookup);
    }

    public List<StartOrderResolver.Resolvable> getImagesConfigsInOrder(QueryService queryService, List<ImageConfiguration> images) {
        return StartOrderResolver.resolve(queryService, this.convertToResolvables(images));
    }

    public PortMapping createPortMapping(RunImageConfiguration runConfig, Properties properties) {
        try {
            return new PortMapping(runConfig.getPorts(), properties);
        }
        catch (IllegalArgumentException exp) {
            throw new IllegalArgumentException("Cannot parse port mapping", exp);
        }
    }

    public void addShutdownHookForStoppingContainers(final boolean keepContainer, final boolean removeVolumes, final boolean removeCustomNetworks) {
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                try {
                    RunService.this.stopStartedContainers(keepContainer, removeVolumes, removeCustomNetworks, null);
                }
                catch (DockerAccessException e) {
                    RunService.this.log.error("Error while stopping containers: %s", e.getMessage());
                }
            }
        });
    }

    private List<StartOrderResolver.Resolvable> convertToResolvables(List<ImageConfiguration> images) {
        ArrayList<StartOrderResolver.Resolvable> ret = new ArrayList<StartOrderResolver.Resolvable>();
        for (ImageConfiguration config : images) {
            if (config.getRunConfiguration().skip()) {
                this.log.info("%s: Skipped running", config.getDescription());
                continue;
            }
            ret.add(config);
        }
        return ret;
    }

    ContainerCreateConfig createContainerConfig(String imageName, RunImageConfiguration runConfig, PortMapping mappedPorts, PomLabel pomLabel, Properties mavenProps) throws DockerAccessException {
        try {
            NetworkConfig networkConfig;
            ContainerCreateConfig config = new ContainerCreateConfig(imageName).hostname(runConfig.getHostname()).domainname(runConfig.getDomainname()).user(runConfig.getUser()).workingDir(runConfig.getWorkingDir()).entrypoint(runConfig.getEntrypoint()).exposedPorts(mappedPorts.getContainerPorts()).environment(runConfig.getEnvPropertyFile(), runConfig.getEnv(), mavenProps).labels(this.mergeLabels(runConfig.getLabels(), pomLabel)).command(runConfig.getCmd()).hostConfig(this.createContainerHostConfig(runConfig, mappedPorts));
            VolumeConfiguration volumeConfig = runConfig.getVolumeConfiguration();
            if (volumeConfig != null) {
                config.binds(volumeConfig.getBind());
            }
            if ((networkConfig = runConfig.getNetworkingConfig()).isCustomNetwork() && networkConfig.hasAliases()) {
                ContainerNetworkingConfig networkingConfig = new ContainerNetworkingConfig().aliases(networkConfig);
                config.networkingConfig(networkingConfig);
            }
            return config;
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException(String.format("Failed to create contained configuration for [%s]", imageName), e);
        }
    }

    private Map<String, String> mergeLabels(Map<String, String> labels, PomLabel runIdLabel) {
        HashMap<String, String> ret = new HashMap<String, String>();
        if (labels != null) {
            ret.putAll(labels);
        }
        if (runIdLabel != null) {
            ret.put(runIdLabel.getKey(), runIdLabel.getValue());
        }
        return ret;
    }

    ContainerHostConfig createContainerHostConfig(RunImageConfiguration runConfig, PortMapping mappedPorts) throws DockerAccessException {
        RestartPolicy restartPolicy = runConfig.getRestartPolicy();
        List<String> links = this.findContainerIdsForLinks(runConfig.getLinks(), runConfig.getNetworkingConfig().isCustomNetwork());
        ContainerHostConfig config = new ContainerHostConfig().extraHosts(runConfig.getExtraHosts()).links(links).portBindings(mappedPorts).privileged(runConfig.getPrivileged()).shmSize(runConfig.getShmSize()).dns(runConfig.getDns()).dnsSearch(runConfig.getDnsSearch()).capAdd(runConfig.getCapAdd()).capDrop(runConfig.getCapDrop()).securityOpts(runConfig.getSecurityOpts()).memory(runConfig.getMemory()).memorySwap(runConfig.getMemorySwap()).restartPolicy(restartPolicy.getName(), restartPolicy.getRetry()).logConfig(runConfig.getLogConfiguration()).ulimits(runConfig.getUlimits());
        this.addVolumeConfig(config, runConfig);
        this.addNetworkingConfig(config, runConfig);
        return config;
    }

    private void addNetworkingConfig(ContainerHostConfig config, RunImageConfiguration runConfig) throws DockerAccessException {
        NetworkConfig networkConfig = runConfig.getNetworkingConfig();
        if (networkConfig.isStandardNetwork()) {
            String alias = networkConfig.getContainerAlias();
            String containerId = alias != null ? this.findContainerId(alias, false) : null;
            config.networkMode(networkConfig.getStandardMode(containerId));
        } else if (networkConfig.isCustomNetwork()) {
            config.networkMode(networkConfig.getCustomNetwork());
        }
    }

    private void addVolumeConfig(ContainerHostConfig config, RunImageConfiguration runConfig) throws DockerAccessException {
        VolumeConfiguration volConfig = runConfig.getVolumeConfiguration();
        if (volConfig != null) {
            config.binds(volConfig.getBind()).volumesFrom(this.findVolumesFromContainers(volConfig.getFrom()));
        }
    }

    private List<String> findContainerIdsForLinks(List<String> links, boolean leaveUnresolvedIfNotFound) throws DockerAccessException {
        ArrayList<String> ret = new ArrayList<String>();
        for (String[] link : EnvUtil.splitOnLastColon(links)) {
            String id = this.findContainerId(link[0], false);
            if (id != null) {
                ret.add(this.queryService.getContainerName(id) + ":" + link[1]);
                continue;
            }
            if (leaveUnresolvedIfNotFound) {
                ret.add(link[0] + ":" + link[1]);
                continue;
            }
            throw new DockerAccessException("No container found for image/alias '%s', unable to link", link[0]);
        }
        return ret.size() != 0 ? ret : null;
    }

    private List<String> findVolumesFromContainers(List<String> images) throws DockerAccessException {
        ArrayList<String> list = new ArrayList<String>();
        if (images != null) {
            for (String image : images) {
                String id = this.findContainerId(image, true);
                if (id == null) {
                    throw new DockerAccessException("No container found for image/alias '%s', unable to mount volumes", image);
                }
                list.add(this.queryService.getContainerName(id));
            }
        }
        return list;
    }

    private String calculateContainerName(String alias, RunImageConfiguration.NamingStrategy namingStrategy) {
        if (namingStrategy == RunImageConfiguration.NamingStrategy.none) {
            return null;
        }
        if (alias == null) {
            throw new IllegalArgumentException("A naming scheme 'alias' requires an image alias to be set");
        }
        return alias;
    }

    private String findContainerId(String imageNameOrAlias, boolean checkAllContainers) throws DockerAccessException {
        Container container;
        String id = this.lookupContainer(imageNameOrAlias);
        if (id == null && (container = this.queryService.getContainer(imageNameOrAlias)) != null && (checkAllContainers || container.isRunning())) {
            id = container.getId();
        }
        return id;
    }

    private void startContainer(ImageConfiguration imageConfig, String id, PomLabel pomLabel) throws DockerAccessException {
        this.log.info("%s: Start container %s", imageConfig.getDescription(), id);
        this.docker.startContainer(id);
        this.tracker.registerContainer(id, imageConfig, pomLabel);
    }

    private void updateMappedPortsAndAddresses(String containerId, PortMapping mappedPorts) throws DockerAccessException {
        Container container = this.queryService.getMandatoryContainer(containerId);
        if (container.isRunning()) {
            mappedPorts.updateProperties(container.getPortBindings());
        } else {
            this.log.warn("Container %s is not running anymore, can not extract dynamic ports", containerId);
        }
    }

    private void shutdown(ContainerTracker.ContainerShutdownDescriptor descriptor, boolean keepContainer, boolean removeVolumes) throws DockerAccessException {
        long waited;
        String containerId = descriptor.getContainerId();
        if (descriptor.getPreStop() != null) {
            try {
                this.execInContainer(containerId, descriptor.getPreStop(), descriptor.getImageConfiguration());
            }
            catch (DockerAccessException e) {
                this.log.error("%s", e.getMessage());
            }
        }
        int killGracePeriod = this.adjustGracePeriod(descriptor.getKillGracePeriod());
        this.log.debug("shutdown will wait max of %d seconds before removing container", killGracePeriod);
        if (killGracePeriod == 0) {
            this.docker.stopContainer(containerId, 0);
            waited = 0L;
        } else {
            waited = this.shutdownAndWait(containerId, killGracePeriod);
        }
        if (!keepContainer) {
            this.removeContainer(descriptor, removeVolumes, containerId);
        }
        this.log.info("%s: Stop%s container %s after %s ms", descriptor.getDescription(), keepContainer ? "" : " and removed", containerId.substring(0, 12), waited);
    }

    public void createCustomNetworkIfNotExistant(String customNetwork) throws DockerAccessException {
        if (!this.queryService.hasNetwork(customNetwork)) {
            this.docker.createNetwork(new NetworkCreateConfig(customNetwork));
        } else {
            this.log.debug("Custom Network " + customNetwork + " found", new Object[0]);
        }
    }

    public void removeCustomNetworks(Collection<Network> networks) throws DockerAccessException {
        for (Network network : networks) {
            this.docker.removeNetwork(network.getId());
        }
    }

    private int adjustGracePeriod(int gracePeriod) {
        int killGracePeriodInSeconds = (gracePeriod + 500) / 1000;
        if (gracePeriod != 0 && killGracePeriodInSeconds == 0) {
            this.log.warn("A kill grace period of %d ms leads to no wait at all since its rounded to seconds. Please use at least 500 as value for wait.kill", gracePeriod);
        }
        return killGracePeriodInSeconds;
    }

    private void removeContainer(ContainerTracker.ContainerShutdownDescriptor descriptor, boolean removeVolumes, String containerId) throws DockerAccessException {
        int shutdownGracePeriod = descriptor.getShutdownGracePeriod();
        if (shutdownGracePeriod != 0) {
            this.log.debug("Shutdown: Wait %d ms before removing container", shutdownGracePeriod);
            WaitUtil.sleep(shutdownGracePeriod);
        }
        this.docker.removeContainer(containerId, removeVolumes);
    }

    private long shutdownAndWait(final String containerId, final int killGracePeriodInSeconds) throws DockerAccessException {
        long waited;
        try {
            waited = WaitUtil.wait(killGracePeriodInSeconds, new Callable<Void>(){

                @Override
                public Void call() throws Exception {
                    RunService.this.docker.stopContainer(containerId, killGracePeriodInSeconds);
                    return null;
                }
            });
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof DockerAccessException) {
                throw (DockerAccessException)e.getCause();
            }
            throw new DockerAccessException(e, "failed to stop container id [%s]", containerId);
        }
        catch (WaitUtil.WaitTimeoutException e) {
            waited = e.getWaited();
            this.log.warn("Stop container id [%s] timed out after %s ms", containerId, waited);
        }
        return waited;
    }
}

