package eventcenter.leveldb.cache;

import eventcenter.leveldb.LevelDBPersistenceAdapter;
import eventcenter.leveldb.PersistenceException;
import org.iq80.leveldb.WriteBatch;

import java.io.IOException;
import java.io.Serializable;
import java.util.*;

/**
 * 使用leveldb作为缓存组件使用
 * @author liumingjian
 * @date 2018/9/8
 **/
public class LevelDBCache {

    final static Map<Integer, Set<String>> cacheStates = Collections.synchronizedMap(new HashMap<Integer, Set<String>>());

    final static String SEPERATOR = "_";

    final String namespace;

    final LevelDBPersistenceAdapter adapter;

    /**
     *
     * @param namespace 名词空间，不同的缓存使用不同的前缀作为缓存的key名，所以这个参数能够做到资源的隔离
     * @param adapter
     */
    public LevelDBCache(String namespace, LevelDBPersistenceAdapter adapter) throws PersistenceException {
        validateCacheStates(namespace, adapter);
        this.namespace = namespace;
        this.adapter = adapter;
    }

    void validateCacheStates(String namespace, LevelDBPersistenceAdapter adapter) throws PersistenceException {
        if(!cacheStates.containsKey(adapter.hashCode())){
            cacheStates.put(adapter.hashCode(), new HashSet<String>());
        }
        if(cacheStates.get(adapter.hashCode()).contains(namespace)){
            throw new PersistenceException("leveldb cache has been contain name:" + namespace);
        }
        cacheStates.get(adapter.hashCode()).add(namespace);
    }

    String buildKey(String key){
        return new StringBuilder(namespace).append(SEPERATOR).append(key).toString();
    }

    /**
     * 获取key的值
     * @param key
     * @param type
     * @param <T>
     * @return
     * @throws PersistenceException
     */
    public <T> T get(String key, Class<T> type) throws PersistenceException {
        return adapter.get(buildKey(key), type);
    }

    /**
     * 设置多个key的值
     * @param key
     * @param value
     * @throws PersistenceException
     */
    public void put(String key, Serializable value) throws PersistenceException {
        adapter.put(buildKey(key), value);
    }

    /**
     *
     * @param keyValues
     * @throws IOException
     * @throws PersistenceException
     */
    public void put(Map<String, Serializable> keyValues) throws IOException, PersistenceException {
        final WriteBatch writeBatch = adapter.getDb().createWriteBatch();
        try{
            final Set<Map.Entry<String, Serializable>> entries = keyValues.entrySet();
            for(Map.Entry<String, Serializable> entry : entries){
                adapter.put(buildKey(entry.getKey()), entry.getValue(), writeBatch);
            }
            adapter.getDb().write(writeBatch);
        }finally {
            writeBatch.close();
        }
    }

    public void remove(String... keys) throws PersistenceException {
        if(null == keys || keys.length == 0){
            return ;
        }
        final String[] ids = new String[keys.length];
        for(int i = 0;i < keys.length;i++){
            ids[i] = buildKey(keys[i]);
        }
        adapter.deleteById(ids);
    }
}
