/*
 * Copyright 2019 etrace.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.etrace.agent.message.metric;

import io.etrace.common.message.ConfigManger;
import io.etrace.common.modal.metric.AbstractMetric;
import io.etrace.common.modal.metric.Metric;
import io.etrace.common.modal.metric.MetricKey;
import io.etrace.common.modal.metric.impl.HistogramImpl;
import io.etrace.common.modal.metric.impl.TimerImpl;

import java.util.HashMap;
import java.util.Map;

public class PackageMetric {
    protected MetricQueue.EventConsumer consumer;
    protected boolean isHistogram;
    protected String topic;
    protected Metric defaultMetric;
    protected Map<MetricKey, Metric> metrics;
    protected long count;
    protected int maxGroup;

    public PackageMetric(ConfigManger configManger, MetricQueue.EventConsumer consumer, Metric metric) {
        if (metric instanceof TimerImpl && ((TimerImpl)metric).isUpperEnable()) {
            isHistogram = true;
            maxGroup = configManger.getMetricConfig().getMaxHistogramGroup();
        } else {
            maxGroup = configManger.getMetricConfig().getMaxGroup();
            isHistogram = false;
        }
        if (metric instanceof AbstractMetric) {
            this.topic = ((AbstractMetric)metric).getTopic();
        }
        this.consumer = consumer;
    }

    private void addTagsMetric(Metric metric) {
        if (metrics == null) {
            metrics = new HashMap<>();
            metrics.put(metric.getTagKey(), buildMetric(metric));
            count++;
            consumer.sendCount++;
            return;
        }
        MetricKey key = metric.getTagKey();
        Metric oldMetric = metrics.get(key);
        if (oldMetric == null) {
            if (count >= maxGroup) {
                return;
            }
            count++;
            consumer.sendCount++;
            metrics.put(key, buildMetric(metric));
        } else {
            oldMetric.merge(metric);
            consumer.mergeCount++;
        }
    }

    private Metric buildMetric(Metric metric) {
        if (isHistogram) {
            HistogramImpl histogram = new HistogramImpl(consumer.getBucketFunction());
            if (metric instanceof TimerImpl) {
                histogram.build((TimerImpl)metric);
                histogram.merge(metric);
            }
            return histogram;
        }
        return metric;
    }

    private void addDefault(Metric metric) {
        if (defaultMetric == null) {
            defaultMetric = buildMetric(metric);
            consumer.sendCount++;
        } else {
            defaultMetric.merge(metric);
            consumer.mergeCount++;
        }
    }

    public void merge(Metric metric) {
        if (metric.getTagKey() == null) {
            addDefault(metric);
        } else {
            addTagsMetric(metric);
        }
    }

    public boolean isEmpty() {
        return (metrics == null || metrics.size() <= 0) && defaultMetric == null;
    }

    public String getTopic() {
        return topic;
    }
}
