/*
 * Copyright 2019 etrace.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.etrace.agent.message.callstack;

import com.google.common.base.Strings;
import com.google.inject.Inject;
import io.etrace.agent.config.AgentConfiguration;
import io.etrace.agent.message.manager.DefaultMessageManager;
import io.etrace.common.Constants;
import io.etrace.common.message.MessageManager;
import io.etrace.common.modal.*;
import io.etrace.common.modal.impl.*;

import java.util.Collections;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

public class CallstackProducer {

    protected final static Event DUMMY_EVENT = new DummyEvent();
    protected final static Transaction DUMMY_TRANSACTION = new DummyTransaction();
    protected final static Heartbeat DUMMY_HEARTBEAT = new DummyHeartbeat();
    private static AtomicLong sequence = new AtomicLong(0);

    protected MessageManager messageManager;

    @Inject
    public CallstackProducer(MessageManager messageManager) {
        this.messageManager = messageManager;
    }

    public void logError(Throwable throwable) {
        logError(null, throwable, null);
    }

    public void logError(String message, Throwable throwable) {
        logError(message, throwable, null);
    }

    public void logError(String message, Throwable throwable, Map<String, String> tags) {
        //need check enabled first
        if (!messageManager.getConfigManager().isEnabled()) {
            return;
        }
        if (!shouldLog(throwable)) {
            return;
        }

        Event event;
        try {
            if (throwable instanceof Error) {
                event = newEvent("Error", throwable.getClass().getName(), message, throwable);
            } else if (throwable instanceof RuntimeException) {
                event = newEvent("RuntimeException", throwable.getClass().getName(), message, throwable);
            } else {
                event = newEvent("Exception", throwable.getClass().getName(), message, throwable);
            }
            event.setStatus("ERROR");
            addTagsToMessage(tags, event);
            event.complete();
        } catch (Throwable ignore) {

        }
    }

    private Event newEvent(String type, String name, String message, Throwable t) {
        if (!messageManager.getConfigManager().isEnabled()) {
            return DUMMY_EVENT;
        }
        if (!messageManager.hasContext()) {
            messageManager.setup();
        }
        Event event = new EventImpl(type, name, message, t, messageManager);
        event.setId(sequence.incrementAndGet());
        return event;
    }

    public void logEvent(String type, String name, String status, String data, Map<String, String> tags) {
        if (!messageManager.getConfigManager().isEnabled()) {
            return;
        }
        Event event = newEvent(type, name, status);
        if (!Strings.isNullOrEmpty(status)) {
            event.setStatus(status);
        }
        event.setData(data);
        addTagsToMessage(tags, event);
        event.complete();
    }

    @Deprecated
    public Event newEvent(String type, String name) {
        return newEvent(type, name, Constants.UNSET);
    }

    private Event newEvent(String type, String name, String status) {
        if (!messageManager.getConfigManager().isEnabled()) {
            return DUMMY_EVENT;
        }
        if (!messageManager.hasContext()) {
            messageManager.setup();
        }
        Event event = new EventImpl(type, name, messageManager);
        event.setId(sequence.incrementAndGet());
        event.setStatus(status);
        return event;
    }

    public Transaction newTransaction(String type, String name) {
        if (!messageManager.getConfigManager().isEnabled()) {
            return DUMMY_TRANSACTION;
        }
        if (!messageManager.hasContext()) {
            messageManager.setup();
        }

        TransactionImpl transaction = new TransactionImpl(type, name, messageManager);

        if (AgentConfiguration.getGlobalTags() != null) {
            for (Map.Entry<String, String> entry : AgentConfiguration.getGlobalTags().entrySet()) {
                transaction.addTag(entry.getKey(), entry.getValue());
            }
        }
        messageManager.startTransaction(transaction);
        transaction.setId(sequence.incrementAndGet());
        return transaction;
    }

    public void logHeartbeat(String type, String name, String status, String data, Map<String, String> tags) {
        if (!messageManager.getConfigManager().isEnabled()) {
            return;
        }
        Heartbeat heartbeat = newHeartbeat(type, name);
        heartbeat.setStatus(status);
        heartbeat.setData(data);
        if (tags != null) {
            heartbeat.addTags(tags);
        }
        heartbeat.complete();
    }

    public Heartbeat newHeartbeat(String type, String name) {
        if (!messageManager.getConfigManager().isEnabled()) {
            return DUMMY_HEARTBEAT;
        }
        if (!messageManager.hasContext()) {
            messageManager.setup();
        }
        Heartbeat heartbeat = new HeartbeatImpl(type, name, messageManager);
        heartbeat.setId(sequence.incrementAndGet());
        addTagsToMessage(Collections.emptyMap(), heartbeat);
        return heartbeat;
    }

    public void shutdown() {
        messageManager.shutdown();
    }

    public String getCurrentRequestId() {
        return messageManager.getCurrentRequestId();
    }

    public String getCurrentRpcIdAndCurrentCall() {
        return messageManager.getCurrentRpcIdAndCurrentCall();
    }

    public String getRpcId() {
        return messageManager.getRpcId();
    }

    public void removeContext() {
        messageManager.removeContext();
    }

    public String nextLocalRpcId() {
        return messageManager.nextLocalRpcId();
    }

    public String nextRemoteRpcId() {
        return messageManager.nextRemoteRpcId();
    }

    public boolean hasContext() {
        return messageManager.hasContext();
    }

    public boolean hasTransaction() {
        return messageManager.hasTransaction();
    }

    public void clean() {
        messageManager.reset();
    }

    public void continueTrace(String requestId, String rpcId) {
        messageManager.setup(requestId, rpcId);
    }

    private boolean shouldLog(Throwable e) {
        return !(messageManager instanceof DefaultMessageManager) || ((DefaultMessageManager)messageManager).shouldLog(
            e);
    }

    public TraceContext exportContext() {
        return messageManager.exportContext();
    }

    public void importContext(TraceContext context) {
        messageManager.importContext(context);
    }

    public boolean isImportContext() {
        return messageManager.isImportContext();
    }

    public String getClientAppId() {
        return messageManager.getClientAppId();
    }

    @Deprecated
    public void redis(String url, String command, long duration, boolean succeed, RedisResponse[] responses,
                      String redisType) {
        // 移除了这里的实现
    }

    private void addTagsToMessage(Map<String, String> tags, Message message) {
        if (tags != null) {
            for (Map.Entry<String, String> entry : tags.entrySet()) {
                message.addTag(entry.getKey(), entry.getValue());
            }
        }
        // global tag has higher priority, may override given tags.
        if (AgentConfiguration.getGlobalTags() != null) {
            for (Map.Entry<String, String> entry : AgentConfiguration.getGlobalTags().entrySet()) {
                message.addTag(entry.getKey(), entry.getValue());
            }
        }
    }

}
