package io.embrace.android.embracesdk.bugshake

import android.app.Activity
import android.app.AlertDialog
import android.content.Intent
import android.graphics.BitmapFactory
import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.View
import android.widget.ArrayAdapter
import android.widget.Button
import android.widget.EditText
import android.widget.ImageView
import android.widget.Spinner
import android.widget.TextView
import io.embrace.android.embracesdk.Embrace
import io.embrace.android.embracesdk.Embrace.BugReportCategory
import io.embrace.android.embracesdk.R

internal class EmbraceBugShakeActivity : Activity() {

    companion object {
        const val MAX_LENGTH = 220
    }

    private lateinit var categorySpinner: Spinner
    private lateinit var reportText: EditText
    private lateinit var screenshotThumbnail: ImageView
    private lateinit var deleteScreenshot: ImageView
    private lateinit var screenshotFrame: View
    private lateinit var categories: Map<String, BugReportCategory>

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        setContentView(R.layout.embrace_shake_activity)

        val sendBtn: Button = findViewById(R.id.embrace_bugshake_send_btn)
        sendBtn.visibility = View.VISIBLE
        sendBtn.setOnClickListener { sendBugReport() }

        categorySpinner = findViewById(R.id.shake_bug_category)
        fillSpinner()

        reportText = findViewById(R.id.embrace_bugshake_bug_report)
        addReportTextListener()

        val screenshot = Embrace.getInstance().getScreenshot()
        screenshot?.let {
            screenshotFrame = findViewById<View>(R.id.embrace_bugshake_screenshot_frame)
            screenshotFrame.visibility = View.VISIBLE

            screenshotThumbnail = findViewById(R.id.embrace_bugshake_screenshot)
            setScreenshotThumbnail(it)
            screenshotThumbnail.setOnClickListener {
                startActivity(Intent(this, EmbraceScreenshotActivity::class.java))
            }

            deleteScreenshot = findViewById(R.id.embrace_bugshake_remove_screenshot)
            deleteScreenshot.setOnClickListener {
                showRemoveScreenshotDialog()
            }
        }
    }

    override fun onBackPressed() {
        if (reportText.text.isNotBlank()) {
            showCloseDialog()
        } else {
            super.onBackPressed()
        }
    }

    private fun sendBugReport() {
        val category = categories[categorySpinner.selectedItem.toString()]
        category?.let {
            val report = reportText.text.toString()
            Embrace.getInstance().logBugReport(category, report)
            reportText.text = null
            showSentDialog()
        }
    }

    private fun showSentDialog() {
        val dialogBuilder = AlertDialog.Builder(this)
        dialogBuilder
            .setMessage(getString(R.string.embrace_bugshake_sent))
            .setCancelable(true)
            .setPositiveButton(getString(R.string.embrace_bugshake_done)) { dialog, _ ->
                dialog.dismiss()
                finish()
            }
        dialogBuilder.create().show()
    }

    private fun showCloseDialog() {
        val dialogBuilder = AlertDialog.Builder(this)
        dialogBuilder
            .setMessage(getString(R.string.embrace_bugshake_close))
            .setNegativeButton(getString(android.R.string.cancel)) { dialog, _ -> dialog.dismiss() }
            .setPositiveButton(getString(android.R.string.ok)) { dialog, _ ->
                dialog.dismiss()
                finish()
            }
        dialogBuilder.create().show()
    }

    private fun showRemoveScreenshotDialog() {
        val dialogBuilder = AlertDialog.Builder(this)
        dialogBuilder
            .setMessage(getString(R.string.embrace_bugshake_remove_screenshot))
            .setNegativeButton(getString(android.R.string.cancel)) { dialog, _ -> dialog.dismiss() }
            .setPositiveButton(getString(android.R.string.ok)) { dialog, _ ->
                dialog.dismiss()
                Embrace.getInstance().deleteScreenshot()
                screenshotFrame.visibility = View.GONE
            }
        dialogBuilder.create().show()
    }

    private fun fillSpinner() {
        categories = mapOf(
            getString(R.string.embrace_bugshake_category_bug) to BugReportCategory.BUG,
            getString(R.string.embrace_bugshake_category_question) to BugReportCategory.QUESTION,
            getString(R.string.embrace_bugshake_category_suggestion) to BugReportCategory.SUGGESTION,
            getString(R.string.embrace_bugshake_category_other) to BugReportCategory.OTHER,
        )
        ArrayAdapter(this, android.R.layout.simple_spinner_item, categories.keys.toTypedArray())
            .also { adapter ->
                adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item)
                categorySpinner.adapter = adapter
            }
    }

    private fun addReportTextListener() {
        val charCount: TextView = findViewById(R.id.embrace_bugshake_bug_char_count)
        charCount.text = getString(R.string.embrace_bugshake_report_characters, 0, MAX_LENGTH)
        reportText.addTextChangedListener(object : TextWatcher {
            override fun afterTextChanged(s: Editable?) {
                val length = reportText.text.toString().length
                if (length > MAX_LENGTH) {
                    reportText.setText(reportText.text.substring(MAX_LENGTH))
                }
                charCount.text = getString(R.string.embrace_bugshake_report_characters, length, MAX_LENGTH)
            }

            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
                // no op
            }

            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
                // no op
            }
        })
    }

    private fun setScreenshotThumbnail(screenshot: ByteArray) {
        try {
            val bitmap = BitmapFactory.decodeByteArray(screenshot, 0, screenshot.size)
            screenshotThumbnail.setImageBitmap(bitmap)
        } catch (ex: Throwable) {
            Embrace.getInstance().logInternalError(ex)
        }
    }
}
