package io.embrace.android.embracesdk.capture.screenshot

import android.app.Activity
import android.graphics.Bitmap
import android.graphics.Canvas
import io.embrace.android.embracesdk.Embrace
import io.embrace.android.embracesdk.clock.Clock
import io.embrace.android.embracesdk.comms.delivery.DeliveryService
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import io.embrace.android.embracesdk.session.ActivityService
import java.io.ByteArrayOutputStream

/**
 * Takes screenshots and submits them to the Embrace API.
 */
internal class EmbraceScreenshotService(
    private val activityService: ActivityService,
    private val deliveryService: DeliveryService,
    private val logger: InternalEmbraceLogger,
    private val clock: Clock
) : ScreenshotService {
    @Volatile
    private var lastScreenshot: Long = 0

    override fun takeScreenshotBugReport(): Boolean {
        synchronized(this) {
            val timestamp = clock.now()
            if (activityService.isInBackground || timestamp - lastScreenshot < SCREENSHOT_COOLDOWN_MS) {
                logger.logDebug("Screenshots are disabled, app is backgrounded, or cooling down")
                return false
            }
            val optionalActivity = activityService.foregroundActivity
            return if (optionalActivity != null) {
                val screenshot = takeAndCompressScreenshot(optionalActivity)
                screenshot?.let {
                    deliveryService.saveScreenshot(it)
                }
                lastScreenshot = timestamp
                if (screenshot == null) {
                    logger.logDebug("Could not take screenshot")
                    return false
                }
                true
            } else {
                logger.logDebug("Screenshot cannot be taken as there is no active activity")
                false
            }
        }
    }

    override fun getScreenshotBugReport(): ByteArray? {
        return deliveryService.loadScreenshot()
    }

    override fun deleteScreenshotBugReport() {
        deliveryService.deleteScreenshot()
    }

    private fun takeScreenshot(activity: Activity?): Bitmap? {
        if (activity == null) {
            Embrace.getInstance()
                .logInternalError("Empty screenshot - activity is null", "Null activity")
            return null
        }

        val view = activity.window.decorView
        if (view == null) {
            Embrace.getInstance().logInternalError(
                "Empty screenshot - decor view is null",
                "activity.getWindow().getDecorView() returned null"
            )
            return null
        }

        return if (view.width > 0 && view.height > 0) {
            try {
                val screenshot =
                    Bitmap.createBitmap(view.width, view.height, Bitmap.Config.ARGB_8888)
                logger.logDeveloper("EmbraceScreenshotService", "Bitmap created")
                view.draw(Canvas(screenshot))
                screenshot
            } catch (e: OutOfMemoryError) {
                Embrace.getInstance().logInternalError(
                    "Empty screenshot - Out of memory drawing screenshot",
                    "Run out of memory while creating the bitmap and drawing the screenshot"
                )
                null
            }
        } else {
            Embrace.getInstance().logInternalError(
                "Empty screenshot - decor view is 0 or lower",
                "Height: " + view.height + ". Width: " + view.width
            )
            null
        }
    }

    private fun takeAndCompressScreenshot(activity: Activity): ByteArray? {
        val screenshot = takeScreenshot(activity)
        if (screenshot == null) {
            Embrace.getInstance()
                .logInternalError("Empty screenshot - null bitmap", "Created bitmap was null")
            return null
        }
        logger.logDebug("Compressing screenshot")
        val stream = ByteArrayOutputStream()
        try {
            screenshot.compress(
                Bitmap.CompressFormat.JPEG,
                SCREENSHOT_JPEG_QUALITY,
                stream
            )
            logger.logDeveloper("EmbraceScreenshotService", "Screenshot compressed")
        } catch (e: OutOfMemoryError) {
            logger.logDebug("Failed to compress screenshot due insufficient memory.", e)
            Embrace.getInstance().logInternalError(
                "Empty screenshot - compress OOM",
                "Failed to compress screenshot. Cause: OOM"
            )
            return null
        }
        screenshot.recycle()
        logger.logDeveloper("EmbraceScreenshotService", "Screenshot recycled")
        return stream.toByteArray()
    }

    companion object {
        private const val SCREENSHOT_COOLDOWN_MS: Long = 2000
        // This value prioritizes a small file size over image quality
        private const val SCREENSHOT_JPEG_QUALITY = 20
    }
}
