@file:JvmName("JVMMediaStreamTrack")

package io.dyte.webrtc

import dev.onvoid.webrtc.media.MediaStreamTrack as NativeMediaStreamTrack
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update

actual abstract class MediaStreamTrack internal constructor(val native: NativeMediaStreamTrack) {

  actual val id: String
    get() = native.id

  actual val kind: MediaStreamTrackKind
    get() =
      when (native.kind) {
        NativeMediaStreamTrack.AUDIO_TRACK_KIND -> MediaStreamTrackKind.Audio
        NativeMediaStreamTrack.VIDEO_TRACK_KIND -> MediaStreamTrackKind.Video
        else -> error("Unknown track kind: ${native.kind}")
      }

  actual val label: String
    get() =
      when (kind) {
        MediaStreamTrackKind.Audio -> "microphone"
        MediaStreamTrackKind.Video -> "camera"
      }

  actual var enabled: Boolean
    get() = native.isEnabled
    set(value) {
      if (value == native.isEnabled) return
      native.isEnabled = value
      onSetEnabled(value)
    }

  private val _state = MutableStateFlow(getInitialState())
  actual val state: StateFlow<MediaStreamTrackState> = _state.asStateFlow()

  //    val constraints: MediaTrackConstraints = MediaTrackConstraints()
  //    val settings: MediaTrackSettings = MediaTrackSettings()

  actual fun stop() {
    if (_state.value is MediaStreamTrackState.Ended) return
    _state.update { MediaStreamTrackState.Ended(it.muted) }
    onStop()
  }

  protected open fun setMuted(muted: Boolean) {
    if (muted) {
      _state.update { it.mute() }
    } else {
      _state.update { it.unmute() }
    }
  }

  protected open fun onSetEnabled(enabled: Boolean) {}

  protected open fun onStop() {}

  private fun getInitialState(): MediaStreamTrackState {
    return when (checkNotNull(native.state)) {
      dev.onvoid.webrtc.media.MediaStreamTrackState.LIVE ->
        MediaStreamTrackState.Live(muted = false)
      dev.onvoid.webrtc.media.MediaStreamTrackState.ENDED ->
        MediaStreamTrackState.Ended(muted = false)
    }
  }
}
