package io.dyte.webrtc

import DyteWebRTC.*
import io.dyte.webrtc.PeerConnectionEvent.ConnectionStateChange
import io.dyte.webrtc.PeerConnectionEvent.IceConnectionStateChange
import io.dyte.webrtc.PeerConnectionEvent.IceGatheringStateChange
import io.dyte.webrtc.PeerConnectionEvent.NegotiationNeeded
import io.dyte.webrtc.PeerConnectionEvent.NewDataChannel
import io.dyte.webrtc.PeerConnectionEvent.NewIceCandidate
import io.dyte.webrtc.PeerConnectionEvent.RemoveTrack
import io.dyte.webrtc.PeerConnectionEvent.RemovedIceCandidates
import io.dyte.webrtc.PeerConnectionEvent.SignalingStateChange
import io.dyte.webrtc.PeerConnectionEvent.StandardizedIceConnectionChange
import io.dyte.webrtc.PeerConnectionEvent.Track
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import platform.Foundation.NSNumber
import platform.darwin.NSObject

actual class PeerConnection actual constructor(rtcConfiguration: RtcConfiguration) :
  NSObject(), DyteRTCPeerConnectionDelegateProtocol {

  val ios: DyteRTCPeerConnection =
    checkNotNull(
      WebRtc.peerConnectionFactory.peerConnectionWithConfiguration(
        configuration = rtcConfiguration.native,
        constraints = DyteRTCMediaConstraints(),
        delegate = this,
      )
    ) {
      "Failed to create peer connection"
    }

  actual val localDescription: SessionDescription?
    get() = ios.localDescription?.asCommon()

  actual val remoteDescription: SessionDescription?
    get() = ios.remoteDescription?.asCommon()

  actual val signalingState: SignalingState
    get() = rtcSignalingStateAsCommon(ios.signalingState())

  actual val iceConnectionState: IceConnectionState
    get() = rtcIceConnectionStateAsCommon(ios.iceConnectionState())

  actual val connectionState: PeerConnectionState
    get() = rtcPeerConnectionStateAsCommon(ios.connectionState())

  actual val iceGatheringState: IceGatheringState
    get() = rtcIceGatheringStateAsCommon(ios.iceGatheringState())

  private val _peerConnectionEvent =
    MutableSharedFlow<PeerConnectionEvent>(extraBufferCapacity = FLOW_BUFFER_CAPACITY)
  internal actual val peerConnectionEvent: Flow<PeerConnectionEvent> =
    _peerConnectionEvent.asSharedFlow()

  private val localTracks = mutableMapOf<String, MediaStreamTrack>()
  private val remoteTracks = mutableMapOf<String, MediaStreamTrack>()

  actual fun createDataChannel(
    label: String,
    id: Int,
    ordered: Boolean,
    maxRetransmitTimeMs: Int,
    maxRetransmits: Int,
    protocol: String,
    negotiated: Boolean,
  ): DataChannel? {
    val config =
      DyteRTCDataChannelConfiguration().also {
        it.channelId = id
        it.isOrdered = ordered
        it.maxRetransmitTimeMs = maxRetransmitTimeMs.toLong()
        it.maxRetransmits = maxRetransmits
        it.protocol = protocol
        it.isNegotiated = negotiated
      }
    return ios.dataChannelForLabel(label, config)?.let { DataChannel(it) }
  }

  actual suspend fun createOffer(options: OfferAnswerOptions): SessionDescription {
    val constraints = options.toRTCMediaConstraints()
    val sessionDescription: DyteRTCSessionDescription =
      ios.awaitResult { offerForConstraints(constraints, it) }
    return sessionDescription.asCommon()
  }

  actual suspend fun createAnswer(options: OfferAnswerOptions): SessionDescription {
    val constraints = options.toRTCMediaConstraints()
    val sessionDescription: DyteRTCSessionDescription =
      ios.awaitResult { answerForConstraints(constraints, it) }
    return sessionDescription.asCommon()
  }

  private fun OfferAnswerOptions.toRTCMediaConstraints(): DyteRTCMediaConstraints {
    val mandatory =
      mutableMapOf<Any?, String?>().apply {
        iceRestart?.let { this += "IceRestart" to "$it" }
        offerToReceiveAudio?.let { this += "OfferToReceiveAudio" to "$it" }
        offerToReceiveVideo?.let { this += "OfferToReceiveVideo" to "$it" }
        voiceActivityDetection?.let { this += "VoiceActivityDetection" to "$it" }
      }
    return DyteRTCMediaConstraints(mandatory, null)
  }

  actual suspend fun setLocalDescription(description: SessionDescription) {
    ios.await { setLocalDescription(description.asIos(), it) }
  }

  actual suspend fun setRemoteDescription(description: SessionDescription) {
    ios.await { setRemoteDescription(description.asIos(), it) }
  }

  actual fun setConfiguration(configuration: RtcConfiguration): Boolean {
    return ios.setConfiguration(configuration.native)
  }

  actual fun addIceCandidate(candidate: IceCandidate): Boolean {
    ios.addIceCandidate(candidate.native)
    return true
  }

  actual fun removeIceCandidates(candidates: List<IceCandidate>): Boolean {
    ios.removeIceCandidates(candidates.map { it.native })
    return true
  }

  actual fun getSenders(): List<RtpSender> =
    ios.senders.map {
      val iosSender = it as DyteRTCRtpSender
      RtpSender(iosSender, localTracks[iosSender.track?.trackId])
    }

  actual fun getReceivers(): List<RtpReceiver> =
    ios.receivers.map {
      val iosReceiver = it as DyteRTCRtpReceiver
      RtpReceiver(iosReceiver, remoteTracks[iosReceiver.track?.trackId])
    }

  actual fun getTransceivers(): List<RtpTransceiver> =
    ios.transceivers.map {
      val iosTransceiver = it as DyteRTCRtpTransceiver
      val senderTrack = localTracks[iosTransceiver.sender.track?.trackId]
      val receiverTrack = remoteTracks[iosTransceiver.receiver.track?.trackId]
      RtpTransceiver(iosTransceiver, senderTrack, receiverTrack)
    }

  actual fun addTrack(track: MediaStreamTrack, vararg streams: MediaStream): RtpSender {
    val streamIds = streams.map { it.id }
    val iosSender = checkNotNull(ios.addTrack(track.ios, streamIds)) { "Failed to add track" }
    localTracks[track.id] = track
    return RtpSender(iosSender, track)
  }

  actual fun removeTrack(sender: RtpSender): Boolean {
    localTracks.remove(sender.track?.id)
    return ios.removeTrack(sender.native)
  }

  actual suspend fun getStats(): RtcStatsReport? {
    return suspendCoroutine { cont ->
      ios.statisticsWithCompletionHandler { stats ->
        stats?.let { cont.resume(RtcStatsReport(stats)) }
      }
    }
  }

  actual suspend fun getSenderStats(sender: RtpSender): RtcStatsReport? {
    return suspendCoroutine { cont ->
      ios.statisticsForSender(sender.native) { stats ->
        stats?.let { cont.resume(RtcStatsReport(stats)) }
      }
    }
  }

  actual suspend fun getReceiverStats(receiver: RtpReceiver): RtcStatsReport? {
    return suspendCoroutine { cont ->
      ios.statisticsForReceiver(receiver.native) { stats ->
        stats?.let { cont.resume(RtcStatsReport(stats)) }
      }
    }
  }

  actual fun close() {
    remoteTracks.values.forEach(MediaStreamTrack::stop)
    remoteTracks.clear()
    ios.close()
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didChangeSignalingState: RTCSignalingState,
  ) {
    val event = SignalingStateChange(rtcSignalingStateAsCommon(didChangeSignalingState))
    _peerConnectionEvent.tryEmit(event)
  }

  @Suppress("CONFLICTING_OVERLOADS", "PARAMETER_NAME_CHANGED_ON_OVERRIDE")
  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didAddStream: DyteRTCMediaStream,
  ) {
    // this deprecated API should not longer be used
    // https://developer.mozilla.org/en-US/docs/Web/API/RTCPeerConnection/onaddstream
  }

  @Suppress("CONFLICTING_OVERLOADS", "PARAMETER_NAME_CHANGED_ON_OVERRIDE")
  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didRemoveStream: DyteRTCMediaStream,
  ) {
    // The removestream event has been removed from the WebRTC specification in favor of
    // the existing removetrack event on the remote MediaStream and the corresponding
    // MediaStream.onremovetrack event handler property of the remote MediaStream.
    // The RTCPeerConnection API is now track-based, so having zero tracks in the remote
    // stream is equivalent to the remote stream being removed and the old removestream event.
    // https://developer.mozilla.org/en-US/docs/Web/API/RTCPeerConnection/onremovestream
  }

  override fun peerConnectionShouldNegotiate(peerConnection: DyteRTCPeerConnection) {
    _peerConnectionEvent.tryEmit(NegotiationNeeded)
  }

  @Suppress("CONFLICTING_OVERLOADS", "PARAMETER_NAME_CHANGED_ON_OVERRIDE")
  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didChangeIceConnectionState: RTCIceConnectionState,
  ) {
    val event = IceConnectionStateChange(rtcIceConnectionStateAsCommon(didChangeIceConnectionState))
    _peerConnectionEvent.tryEmit(event)
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didChangeIceGatheringState: RTCIceGatheringState,
  ) {
    val event = IceGatheringStateChange(rtcIceGatheringStateAsCommon(didChangeIceGatheringState))
    _peerConnectionEvent.tryEmit(event)
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didGenerateIceCandidate: DyteRTCIceCandidate,
  ) {
    val event = NewIceCandidate(IceCandidate(didGenerateIceCandidate))
    _peerConnectionEvent.tryEmit(event)
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didRemoveIceCandidates: List<*>,
  ) {
    val candidates = didRemoveIceCandidates.map { IceCandidate(it as DyteRTCIceCandidate) }
    val event = RemovedIceCandidates(candidates)
    _peerConnectionEvent.tryEmit(event)
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didOpenDataChannel: DyteRTCDataChannel,
  ) {
    val event = NewDataChannel(DataChannel(didOpenDataChannel))
    _peerConnectionEvent.tryEmit(event)
  }

  @Suppress("CONFLICTING_OVERLOADS", "PARAMETER_NAME_CHANGED_ON_OVERRIDE")
  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didChangeStandardizedIceConnectionState: RTCIceConnectionState,
  ) {
    val event =
      StandardizedIceConnectionChange(
        rtcIceConnectionStateAsCommon(didChangeStandardizedIceConnectionState)
      )
    _peerConnectionEvent.tryEmit(event)
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didChangeConnectionState: RTCPeerConnectionState,
  ) {
    val event = ConnectionStateChange(rtcPeerConnectionStateAsCommon(didChangeConnectionState))
    _peerConnectionEvent.tryEmit(event)
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didAddReceiver: DyteRTCRtpReceiver,
    streams: List<*>,
  ) {
    val transceiver =
      ios.transceivers
        .map { it as DyteRTCRtpTransceiver }
        .find { it.receiver.receiverId == didAddReceiver.receiverId } ?: return
    val mid = transceiver.mid

    val iosStreams = streams.map { it as DyteRTCMediaStream }

    val audioTracks =
      iosStreams
        .flatMap { it.audioTracks }
        .map { it as DyteRTCAudioTrack }
        .map { remoteTracks.getOrPut(it.trackId) { AudioStreamTrack(it) } }

    val videoTracks =
      iosStreams
        .flatMap { it.videoTracks }
        .map { it as DyteRTCVideoTrack }
        .map { remoteTracks.getOrPut(it.trackId) { VideoStreamTrack(it) } }

    val commonStreams =
      iosStreams.map { iosStream ->
        MediaStream(ios = iosStream, id = iosStream.streamId).also { stream ->
          audioTracks.forEach(stream::addTrack)
          videoTracks.forEach(stream::addTrack)
        }
      }

    val receiverTrack = remoteTracks[didAddReceiver.track?.trackId]
    val senderTrack = localTracks[transceiver.sender.track?.trackId]

    val trackEvent =
      TrackEvent(
        receiver = RtpReceiver(didAddReceiver, receiverTrack),
        streams = commonStreams,
        track = receiverTrack,
        transceiver = RtpTransceiver(transceiver, senderTrack, receiverTrack),
        mid = mid,
      )

    val event = Track(trackEvent)
    _peerConnectionEvent.tryEmit(event)
  }

  override fun peerConnection(
    peerConnection: DyteRTCPeerConnection,
    didRemoveReceiver: DyteRTCRtpReceiver,
  ) {
    val track = remoteTracks.remove(didRemoveReceiver.track?.trackId)
    val event = RemoveTrack(RtpReceiver(didRemoveReceiver, track))
    _peerConnectionEvent.tryEmit(event)
    track?.stop()
  }

  actual fun addTransceiver(
    track: MediaStreamTrack?,
    kind: MediaStreamTrackKind?,
    init: CommonRtpTransceiverInit,
  ): RtpTransceiver? {
    val nativeRtpTransceiverInit = DyteRTCRtpTransceiverInit()
    val nativeEncoding = mutableListOf<DyteRTCRtpEncodingParameters>()
    init.sendEncodings.forEach {
      val n = DyteRTCRtpEncodingParameters()
      if (it.rid != null) n.setRid(it.rid)
      if (it.scaleResolutionDownBy != null)
        n.setScaleResolutionDownBy(NSNumber(it.scaleResolutionDownBy!!))
      n.setIsActive(it.active)
      nativeEncoding.add(n)
    }
    nativeRtpTransceiverInit.setDirection(init.direction.asNative())
    nativeRtpTransceiverInit.setStreamIds(init.streams.map { it.id })
    nativeRtpTransceiverInit.setSendEncodings(nativeEncoding.toList())

    var transceiver: RtpTransceiver? = null

    if (track != null) {
      val trans = ios.addTransceiverWithTrack(track.ios, nativeRtpTransceiverInit)

      if (trans != null) {
        val senderTrack = localTracks[trans.sender.track?.trackId]
        val receiverTrack = remoteTracks[trans.receiver.track?.trackId]

        transceiver = RtpTransceiver(trans, senderTrack, receiverTrack)
      }
    } else if (kind != null) {
      var nativeKind = RTCRtpMediaType.RTCRtpMediaTypeVideo
      if (kind == MediaStreamTrackKind.Audio) {
        nativeKind = RTCRtpMediaType.RTCRtpMediaTypeAudio
      }
      val trans = ios.addTransceiverOfType(nativeKind, nativeRtpTransceiverInit)

      if (trans != null) {
        val senderTrack = localTracks[trans.sender.track?.trackId]
        val receiverTrack = remoteTracks[trans.receiver.track?.trackId]

        transceiver = RtpTransceiver(trans, senderTrack, receiverTrack)
      }
    }

    return transceiver
  }
}
