@file:JvmName("WebRtcKmpAndroid")

package io.dyte.webrtc

import io.webrtc.DefaultVideoDecoderFactory
import io.webrtc.DefaultVideoEncoderFactory
import io.webrtc.EglBase
import io.webrtc.PeerConnectionFactory
import io.webrtc.SoftwareVideoDecoderFactory
import io.webrtc.SoftwareVideoEncoderFactory

actual object WebRtc {

  private var _eglBase: EglBase? = null
  val rootEglBase: EglBase
    get() =
      checkNotNull(_eglBase) { "Ensure WebRtc.initialize() is called before accessing this field!" }

  private var _peerConnectionFactory: PeerConnectionFactory? = null
  val peerConnectionFactory: PeerConnectionFactory
    get() =
      checkNotNull(_peerConnectionFactory) {
        "Ensure WebRtc.initialize() is called before accessing this field!"
      }

  actual fun initialize(forceSoftwareEncoder: Boolean, forceSoftwareDecoder: Boolean) {
    check(_peerConnectionFactory == null) { "WebRtc already initialized." }
    _eglBase = EglBase.create()
    initializePeerConnectionFactory()
    _peerConnectionFactory =
      getDefaultPeerConnectionBuilder(forceSoftwareEncoder, forceSoftwareDecoder)
        .createPeerConnectionFactory()
  }

  private fun initializePeerConnectionFactory() {
    val fieldTrialsString =
      emptyMap<String, String>().entries.joinToString(separator = "/") { "${it.key}/${it.value}" }

    val initOptions =
      PeerConnectionFactory.InitializationOptions.builder(ApplicationContextHolder.context)
        .setFieldTrials(fieldTrialsString)
        .setEnableInternalTracer(false)
        .createInitializationOptions()

    PeerConnectionFactory.initialize(initOptions)
  }

  private fun getDefaultPeerConnectionBuilder(
    forceSoftwareEncoder: Boolean,
    forceSoftwareDecoder: Boolean
  ): PeerConnectionFactory.Builder {
    val videoEncoderFactory =
      if (forceSoftwareEncoder) SoftwareVideoEncoderFactory()
      else DefaultVideoEncoderFactory(rootEglBase.eglBaseContext, true, false)

    val videoDecoderFactory =
      if (forceSoftwareDecoder) SoftwareVideoDecoderFactory()
      else DefaultVideoDecoderFactory(rootEglBase.eglBaseContext)

    return PeerConnectionFactory.builder()
      .setVideoEncoderFactory(videoEncoderFactory)
      .setVideoDecoderFactory(videoDecoderFactory)
  }

  actual fun dispose() {
    if (_peerConnectionFactory == null) return

    _eglBase?.release()
    _eglBase = null

    _peerConnectionFactory?.dispose()
    _peerConnectionFactory = null

    PeerConnectionFactory.shutdownInternalTracer()
  }
}

@Deprecated("Use WebRtc.rootEglBase", replaceWith = ReplaceWith("WebRtc.rootEglBase"))
val eglBase: EglBase
  get() = WebRtc.rootEglBase

@Deprecated(
  "Use WebRtc.rootEglBase.eglBaseContext",
  replaceWith = ReplaceWith("WebRtc.rootEglBase.eglBaseContext")
)
val eglBaseContext: EglBase.Context
  get() = WebRtc.rootEglBase.eglBaseContext
