package io.dyte.webrtc

import io.dyte.webrtc.PeerConnectionEvent.ConnectionStateChange
import io.dyte.webrtc.PeerConnectionEvent.IceConnectionStateChange
import io.dyte.webrtc.PeerConnectionEvent.IceGatheringStateChange
import io.dyte.webrtc.PeerConnectionEvent.NegotiationNeeded
import io.dyte.webrtc.PeerConnectionEvent.NewDataChannel
import io.dyte.webrtc.PeerConnectionEvent.NewIceCandidate
import io.dyte.webrtc.PeerConnectionEvent.RemoveTrack
import io.dyte.webrtc.PeerConnectionEvent.RemovedIceCandidates
import io.dyte.webrtc.PeerConnectionEvent.SignalingStateChange
import io.dyte.webrtc.PeerConnectionEvent.StandardizedIceConnectionChange
import io.dyte.webrtc.PeerConnectionEvent.Track
import io.webrtc.CandidatePairChangeEvent
import io.webrtc.DataChannel as AndroidDataChannel
import io.webrtc.IceCandidate as AndroidIceCandidate
import io.webrtc.MediaConstraints
import io.webrtc.MediaStream as AndroidMediaStream
import io.webrtc.MediaStreamTrack.MediaType
import io.webrtc.PeerConnection as AndroidPeerConnection
import io.webrtc.RtpParameters.Encoding
import io.webrtc.RtpReceiver as AndroidRtpReceiver
import io.webrtc.RtpTransceiver.RtpTransceiverInit
import io.webrtc.SdpObserver
import io.webrtc.SessionDescription as AndroidSessionDescription
import kotlin.coroutines.Continuation
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.coroutines.suspendCoroutine
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow

actual class PeerConnection actual constructor(rtcConfiguration: RtcConfiguration) {

  val android: AndroidPeerConnection =
    WebRtc.peerConnectionFactory.createPeerConnection(
      rtcConfiguration.android,
      AndroidPeerConnectionObserver()
    )
      ?: error("Creating PeerConnection failed")

  actual val localDescription: SessionDescription?
    get() = android.localDescription?.asCommon()

  actual val remoteDescription: SessionDescription?
    get() = android.remoteDescription?.asCommon()

  actual val signalingState: SignalingState
    get() = android.signalingState().asCommon()

  actual val iceConnectionState: IceConnectionState
    get() = android.iceConnectionState().asCommon()

  actual val connectionState: PeerConnectionState
    get() = android.connectionState().asCommon()

  actual val iceGatheringState: IceGatheringState
    get() = android.iceGatheringState().asCommon()

  private val _peerConnectionEvent =
    MutableSharedFlow<PeerConnectionEvent>(extraBufferCapacity = FLOW_BUFFER_CAPACITY)
  internal actual val peerConnectionEvent: Flow<PeerConnectionEvent> =
    _peerConnectionEvent.asSharedFlow()

  private val localTracks = mutableMapOf<String, MediaStreamTrack>()
  private val remoteTracks = mutableMapOf<String, MediaStreamTrack>()

  actual fun createDataChannel(
    label: String,
    id: Int,
    ordered: Boolean,
    maxRetransmitTimeMs: Int,
    maxRetransmits: Int,
    protocol: String,
    negotiated: Boolean
  ): DataChannel? {
    val init =
      AndroidDataChannel.Init().also {
        it.id = id
        it.ordered = ordered
        it.maxRetransmitTimeMs = maxRetransmitTimeMs
        it.maxRetransmits = maxRetransmits
        it.protocol = protocol
        it.negotiated = negotiated
      }
    return android.createDataChannel(label, init)?.let { DataChannel(it) }
  }

  actual suspend fun createOffer(options: OfferAnswerOptions): SessionDescription {
    return suspendCoroutine { cont ->
      android.createOffer(createSdpObserver(cont), options.toMediaConstraints())
    }
  }

  actual suspend fun createAnswer(options: OfferAnswerOptions): SessionDescription {
    return suspendCoroutine { cont ->
      android.createAnswer(createSdpObserver(cont), options.toMediaConstraints())
    }
  }

  private fun OfferAnswerOptions.toMediaConstraints(): MediaConstraints {
    return MediaConstraints().apply {
      iceRestart?.let { mandatory += MediaConstraints.KeyValuePair("IceRestart", "$it") }
      offerToReceiveAudio?.let {
        mandatory += MediaConstraints.KeyValuePair("OfferToReceiveAudio", "$it")
      }
      offerToReceiveVideo?.let {
        mandatory += MediaConstraints.KeyValuePair("OfferToReceiveVideo", "$it")
      }
      voiceActivityDetection?.let {
        mandatory += MediaConstraints.KeyValuePair("VoiceActivityDetection", "$it")
      }
    }
  }

  private fun createSdpObserver(continuation: Continuation<SessionDescription>): SdpObserver {
    return object : SdpObserver {
      override fun onCreateSuccess(description: AndroidSessionDescription) {
        continuation.resume(description.asCommon())
      }

      override fun onSetSuccess() {
        // not applicable for creating SDP
      }

      override fun onCreateFailure(error: String?) {
        continuation.resumeWithException(RuntimeException("Creating SDP failed: $error"))
      }

      override fun onSetFailure(error: String?) {
        // not applicable for creating SDP
      }
    }
  }

  actual suspend fun setLocalDescription(description: SessionDescription) {
    return suspendCoroutine {
      android.setLocalDescription(setSdpObserver(it), description.asAndroid())
    }
  }

  actual suspend fun setRemoteDescription(description: SessionDescription) {
    return suspendCoroutine {
      android.setRemoteDescription(setSdpObserver(it), description.asAndroid())
    }
  }

  private fun setSdpObserver(continuation: Continuation<Unit>): SdpObserver {
    return object : SdpObserver {
      override fun onCreateSuccess(description: AndroidSessionDescription) {
        // not applicable for setting SDP
      }

      override fun onSetSuccess() {
        continuation.resume(Unit)
      }

      override fun onCreateFailure(error: String?) {
        // not applicable for setting SDP
      }

      override fun onSetFailure(error: String?) {
        continuation.resumeWithException(RuntimeException("Setting SDP failed: $error"))
      }
    }
  }

  actual fun setConfiguration(configuration: RtcConfiguration): Boolean {
    return android.setConfiguration(configuration.android)
  }

  actual fun addIceCandidate(candidate: IceCandidate): Boolean {
    return android.addIceCandidate(candidate.native)
  }

  actual fun removeIceCandidates(candidates: List<IceCandidate>): Boolean {
    return android.removeIceCandidates(candidates.map { it.native }.toTypedArray())
  }

  actual fun getSenders(): List<RtpSender> =
    android.senders.map { RtpSender(it, localTracks[it.track()?.id()]) }

  actual fun getReceivers(): List<RtpReceiver> =
    android.receivers.map { RtpReceiver(it, remoteTracks[it.track()?.id()]) }

  actual fun getTransceivers(): List<RtpTransceiver> =
    android.transceivers.map {
      val senderTrack = localTracks[it.sender.track()?.id()]
      val receiverTrack = remoteTracks[it.receiver.track()?.id()]
      RtpTransceiver(it, senderTrack, receiverTrack)
    }

  actual fun addTrack(track: MediaStreamTrack, vararg streams: MediaStream): RtpSender {
    val streamIds = streams.map { it.id }
    localTracks[track.id] = track
    return RtpSender(android.addTrack(track.android, streamIds), track)
  }

  actual fun removeTrack(sender: RtpSender): Boolean {
    localTracks.remove(sender.track?.id)
    return android.removeTrack(sender.native)
  }

  actual suspend fun getStats(): RtcStatsReport? {
    return suspendCoroutine { cont -> android.getStats { cont.resume(RtcStatsReport(it)) } }
  }

  actual fun close() {
    remoteTracks.values.forEach(MediaStreamTrack::stop)
    remoteTracks.clear()
    android.dispose()
  }

  internal inner class AndroidPeerConnectionObserver : AndroidPeerConnection.Observer {
    override fun onSignalingChange(newState: AndroidPeerConnection.SignalingState) {
      _peerConnectionEvent.tryEmit(SignalingStateChange(newState.asCommon()))
    }

    override fun onIceConnectionChange(newState: AndroidPeerConnection.IceConnectionState) {
      _peerConnectionEvent.tryEmit(IceConnectionStateChange(newState.asCommon()))
    }

    override fun onStandardizedIceConnectionChange(
      newState: AndroidPeerConnection.IceConnectionState
    ) {
      _peerConnectionEvent.tryEmit(StandardizedIceConnectionChange(newState.asCommon()))
    }

    override fun onConnectionChange(newState: AndroidPeerConnection.PeerConnectionState) {
      _peerConnectionEvent.tryEmit(ConnectionStateChange(newState.asCommon()))
    }

    override fun onIceConnectionReceivingChange(receiving: Boolean) {}

    override fun onIceGatheringChange(newState: AndroidPeerConnection.IceGatheringState) {
      _peerConnectionEvent.tryEmit(IceGatheringStateChange(newState.asCommon()))
    }

    override fun onIceCandidate(candidate: AndroidIceCandidate) {
      _peerConnectionEvent.tryEmit(NewIceCandidate(IceCandidate(candidate)))
    }

    override fun onIceCandidatesRemoved(candidates: Array<out AndroidIceCandidate>) {
      _peerConnectionEvent.tryEmit(RemovedIceCandidates(candidates.map { IceCandidate(it) }))
    }

    override fun onAddStream(nativeStream: AndroidMediaStream) {
      // this deprecated API should not longer be used
      // https://developer.mozilla.org/en-US/docs/Web/API/RTCPeerConnection/onaddstream
    }

    override fun onRemoveStream(nativeStream: AndroidMediaStream) {
      // The removestream event has been removed from the WebRTC specification in favor of
      // the existing removetrack event on the remote MediaStream and the corresponding
      // MediaStream.onremovetrack event handler property of the remote MediaStream.
      // The RTCPeerConnection API is now track-based, so having zero tracks in the remote
      // stream is equivalent to the remote stream being removed and the old removestream event.
      // https://developer.mozilla.org/en-US/docs/Web/API/RTCPeerConnection/onremovestream
    }

    override fun onDataChannel(dataChannel: AndroidDataChannel) {
      _peerConnectionEvent.tryEmit(NewDataChannel(DataChannel(dataChannel)))
    }

    override fun onRenegotiationNeeded() {
      _peerConnectionEvent.tryEmit(NegotiationNeeded)
    }

    override fun onAddTrack(
      receiver: AndroidRtpReceiver,
      androidStreams: Array<out AndroidMediaStream>
    ) {
      val transceiver = android.transceivers.find { it.receiver.id() == receiver.id() } ?: return

      val audioTracks =
        androidStreams
          .flatMap { it.audioTracks }
          .map { remoteTracks.getOrPut(it.id()) { AudioStreamTrack(it) } }

      val videoTracks =
        androidStreams
          .flatMap { it.videoTracks }
          .map { remoteTracks.getOrPut(it.id()) { VideoStreamTrack(it) } }

      val streams =
        androidStreams.map { androidStream ->
          MediaStream(
              android = androidStream,
              id = androidStream.id,
            )
            .also { stream ->
              audioTracks.forEach(stream::addTrack)
              videoTracks.forEach(stream::addTrack)
            }
        }

      val senderTrack = localTracks[transceiver.sender.track()?.id()]
      val receiverTrack = remoteTracks[receiver.track()?.id()]

      val trackEvent =
        TrackEvent(
          receiver = RtpReceiver(receiver, receiverTrack),
          streams = streams,
          track = receiverTrack,
          transceiver = RtpTransceiver(transceiver, senderTrack, receiverTrack)
        )

      _peerConnectionEvent.tryEmit(Track(trackEvent))
    }

    override fun onRemoveTrack(receiver: AndroidRtpReceiver) {
      val track = remoteTracks.remove(receiver.track()?.id())
      _peerConnectionEvent.tryEmit(RemoveTrack(RtpReceiver(receiver, track)))
      track?.stop()
    }

    override fun onSelectedCandidatePairChanged(event: CandidatePairChangeEvent) {
      // not implemented
    }
  }

  actual fun addTransceiver(
    track: MediaStreamTrack?,
    kind: MediaStreamTrackKind?,
    init: CommonRtpTransceiverInit
  ): RtpTransceiver? {
    val nativeRtpTransceiverInit =
      RtpTransceiverInit(
        init.direction.asNative(),
        init.streams.map { it.id },
        init.sendEncodings.map { Encoding(it.rid, it.active, it.scaleResolutionDownBy) }
      )

    var transceiver: RtpTransceiver? = null

    if (track != null) {
      val trans = android.addTransceiver(track.android, nativeRtpTransceiverInit)
      val senderTrack = localTracks[trans.sender.track()?.id()]
      val receiverTrack = remoteTracks[trans.receiver.track()?.id()]

      transceiver = RtpTransceiver(trans, senderTrack, receiverTrack)
    } else if (kind != null) {
      var nativeKind = MediaType.MEDIA_TYPE_VIDEO
      if (kind == MediaStreamTrackKind.Audio) {
        nativeKind = MediaType.MEDIA_TYPE_AUDIO
      }
      val trans = android.addTransceiver(nativeKind, nativeRtpTransceiverInit)
      val senderTrack = localTracks[trans.sender.track()?.id()]
      val receiverTrack = remoteTracks[trans.receiver.track()?.id()]

      transceiver = RtpTransceiver(trans, senderTrack, receiverTrack)
    }

    return transceiver
  }
}

private fun AndroidPeerConnection.SignalingState.asCommon(): SignalingState {
  return when (this) {
    AndroidPeerConnection.SignalingState.STABLE -> SignalingState.Stable
    AndroidPeerConnection.SignalingState.HAVE_LOCAL_OFFER -> SignalingState.HaveLocalOffer
    AndroidPeerConnection.SignalingState.HAVE_LOCAL_PRANSWER -> SignalingState.HaveLocalPranswer
    AndroidPeerConnection.SignalingState.HAVE_REMOTE_OFFER -> SignalingState.HaveRemoteOffer
    AndroidPeerConnection.SignalingState.HAVE_REMOTE_PRANSWER -> SignalingState.HaveRemotePranswer
    AndroidPeerConnection.SignalingState.CLOSED -> SignalingState.Closed
  }
}

private fun AndroidPeerConnection.IceConnectionState.asCommon(): IceConnectionState {
  return when (this) {
    AndroidPeerConnection.IceConnectionState.NEW -> IceConnectionState.New
    AndroidPeerConnection.IceConnectionState.CHECKING -> IceConnectionState.Checking
    AndroidPeerConnection.IceConnectionState.CONNECTED -> IceConnectionState.Connected
    AndroidPeerConnection.IceConnectionState.COMPLETED -> IceConnectionState.Completed
    AndroidPeerConnection.IceConnectionState.FAILED -> IceConnectionState.Failed
    AndroidPeerConnection.IceConnectionState.DISCONNECTED -> IceConnectionState.Disconnected
    AndroidPeerConnection.IceConnectionState.CLOSED -> IceConnectionState.Closed
  }
}

private fun AndroidPeerConnection.PeerConnectionState.asCommon(): PeerConnectionState {
  return when (this) {
    AndroidPeerConnection.PeerConnectionState.NEW -> PeerConnectionState.New
    AndroidPeerConnection.PeerConnectionState.CONNECTING -> PeerConnectionState.Connecting
    AndroidPeerConnection.PeerConnectionState.CONNECTED -> PeerConnectionState.Connected
    AndroidPeerConnection.PeerConnectionState.DISCONNECTED -> PeerConnectionState.Disconnected
    AndroidPeerConnection.PeerConnectionState.FAILED -> PeerConnectionState.Failed
    AndroidPeerConnection.PeerConnectionState.CLOSED -> PeerConnectionState.Closed
  }
}

private fun AndroidPeerConnection.IceGatheringState.asCommon(): IceGatheringState {
  return when (this) {
    AndroidPeerConnection.IceGatheringState.NEW -> IceGatheringState.New
    AndroidPeerConnection.IceGatheringState.GATHERING -> IceGatheringState.Gathering
    AndroidPeerConnection.IceGatheringState.COMPLETE -> IceGatheringState.Complete
  }
}
