@file:JvmName("AndroidMediaDevices")

package io.dyte.webrtc

import android.Manifest
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.Matrix
import android.opengl.GLES20
import android.opengl.GLUtils
import androidx.core.content.ContextCompat
import io.dyte.webrtc.yuv.YuvFrame
import java.util.UUID
import org.webrtc.Camera2Enumerator
import org.webrtc.MediaConstraints
import org.webrtc.SurfaceTextureHelper
import org.webrtc.TextureBufferImpl
import org.webrtc.VideoFrame
import org.webrtc.VideoProcessor
import org.webrtc.VideoSink
import org.webrtc.YuvConverter

internal actual val mediaDevices: MediaDevices = MediaDevicesImpl

private object MediaDevicesImpl : MediaDevices {

  override suspend fun getUserMedia(
    streamConstraints: MediaStreamConstraintsBuilder.() -> Unit
  ): MediaStream {
    val constraints =
      MediaStreamConstraintsBuilder().let {
        streamConstraints(it)
        it.constraints
      }

    var audioTrack: AudioStreamTrack? = null
    if (constraints.audio != null) {
      checkRecordAudioPermission()
      val mediaConstraints =
        MediaConstraints().apply {
          mandatory.addAll(
            constraints.audio.toMandatoryMap().map { (k, v) ->
              MediaConstraints.KeyValuePair("$k", "$v")
            }
          )
          optional.addAll(
            constraints.audio.toOptionalMap().map { (k, v) ->
              MediaConstraints.KeyValuePair("$k", "$v")
            }
          )
        }
      val audioSource = WebRtc.peerConnectionFactory.createAudioSource(mediaConstraints)
      val androidTrack =
        WebRtc.peerConnectionFactory.createAudioTrack(UUID.randomUUID().toString(), audioSource)
      audioTrack = AudioStreamTrack(androidTrack, audioSource)
    }

    var videoTrack: VideoStreamTrack? = null
    var newFrame: VideoFrame? = null

    if (constraints.video != null) {
      checkCameraPermission()

      val videoSource = WebRtc.peerConnectionFactory.createVideoSource(false)

      videoSource.setVideoProcessor(
        object : VideoProcessor {
          var _sink: VideoSink? = null
          val textureHelper =
            SurfaceTextureHelper.create("CaptureThread", WebRtc.rootEglBase.eglBaseContext)

          override fun onCapturerStarted(p0: Boolean) {}

          override fun onCapturerStopped() {}

          override fun onFrameCaptured(p0: VideoFrame?) {
            var skipFrame = false

            if (AndroidVideoMiddlewares.getMiddlewares().isNotEmpty()) {
              val bitmap = YuvFrame(p0, YuvFrame.PROCESSING_NONE, p0!!.timestampNs).bitmap!!

              DyteVideoFrame.FRAME_HEIGHT = bitmap.height
              DyteVideoFrame.FRAME_WIDTH = bitmap.width

              var middlewareFrame: DyteVideoFrame? = DyteVideoFrame.fromBitmap(bitmap)

              run breaking@{
                AndroidVideoMiddlewares.getMiddlewares().forEach { mw ->
                  middlewareFrame = mw.onVideoFrame(middlewareFrame!!)
                  if (middlewareFrame == null) {
                    return@breaking
                  }
                }
              }

              textureHelper.handler.post {
                if (middlewareFrame == null) {
                  skipFrame = true
                } else {
                  newFrame =
                    bitmapToVideoFrame(middlewareFrame!!.toBitmap(), p0.timestampNs, textureHelper)
                }
                if (!skipFrame) _sink?.onFrame(newFrame)
              }
            } else {
              newFrame = p0
              if (!skipFrame) _sink?.onFrame(newFrame)
            }
          }

          override fun setSink(p0: VideoSink?) {
            _sink = p0
          }
        }
      )

      val videoCaptureController = CameraVideoCaptureController(constraints.video, videoSource)
      val androidTrack =
        WebRtc.peerConnectionFactory.createVideoTrack(UUID.randomUUID().toString(), videoSource)
      videoTrack = VideoStreamTrack(androidTrack, videoCaptureController)
    }

    val localMediaStream =
      WebRtc.peerConnectionFactory.createLocalMediaStream(UUID.randomUUID().toString())
    return MediaStream(localMediaStream).apply {
      if (audioTrack != null) addTrack(audioTrack)
      if (videoTrack != null) addTrack(videoTrack)
    }
  }

  override suspend fun getDisplayMedia(): MediaStream {
    TODO("Not yet implemented for Android platform")
  }

  override suspend fun supportsDisplayMedia(): Boolean = false

  private fun checkRecordAudioPermission() {
    val result =
      ContextCompat.checkSelfPermission(
        ApplicationContextHolder.context,
        Manifest.permission.RECORD_AUDIO
      )
    if (result == PackageManager.PERMISSION_DENIED) throw RecordAudioPermissionException()
  }

  private fun checkCameraPermission() {
    val result =
      ContextCompat.checkSelfPermission(
        ApplicationContextHolder.context,
        Manifest.permission.CAMERA
      )
    if (result == PackageManager.PERMISSION_DENIED) throw CameraPermissionException()
  }

  override suspend fun enumerateDevices(): List<MediaDeviceInfo> {
    val enumerator = Camera2Enumerator(ApplicationContextHolder.context)
    return enumerator.deviceNames.map {
      MediaDeviceInfo(deviceId = it, label = it, kind = MediaDeviceKind.VideoInput)
    }
  }

  private fun bitmapToVideoFrame(
    bitmap: Bitmap,
    timestampNs: Long,
    textureHelper: SurfaceTextureHelper
  ): VideoFrame {
    val converter = YuvConverter()

    val textures = IntArray(1)
    GLES20.glGenTextures(0, textures, 0)

    val matrix = Matrix()
    matrix.setScale(-1f, 1f)

    val textureBuffer =
      TextureBufferImpl(
        bitmap.width,
        bitmap.height,
        VideoFrame.TextureBuffer.Type.RGB,
        textures[0],
        matrix,
        textureHelper.handler,
        converter,
        null
      )

    GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_MIN_FILTER, GLES20.GL_NEAREST)
    GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_MAG_FILTER, GLES20.GL_NEAREST)
    GLUtils.texImage2D(GLES20.GL_TEXTURE_2D, 0, bitmap, 0)

    val opBuffer = textureBuffer.toI420()

    return VideoFrame(opBuffer, 180, timestampNs)
  }
}
