package io.dyte.sockrates.client.backoff

import kotlin.math.floor
import kotlin.math.min
import kotlin.math.pow
import kotlin.random.Random

interface Backoff {
  val attemptsTillNow: Int

  fun nextBackoffMillis(): Long

  fun reset()
}

class ExponentialBackoff(
  private val initialIntervalMillis: Long = 1000L,
  private val maxIntervalMillis: Long = 10000L,
  private val addJitter: Boolean = false,
) : Backoff {
  private var currentAttempt: Int = 0

  override fun reset() {
    currentAttempt = 0
  }

  override val attemptsTillNow: Int
    get() = currentAttempt

  override fun nextBackoffMillis(): Long {
    // (2^attemptNumber * initialInterval)
    val exponentialBackoff = ((2.0).pow(currentAttempt) * initialIntervalMillis).toLong()

    currentAttempt++

    return if (addJitter) {
      // random_between(0, min(maxInterval, (2^attemptNumber * initialInterval)
      randomLongInRange(0, min(maxIntervalMillis, exponentialBackoff))
    } else {
      min(maxIntervalMillis, exponentialBackoff)
    }
  }

  companion object {
    private fun randomLongInRange(lower: Long, upper: Long): Long {
      return (floor((Random.nextFloat() * (upper - lower + 1)) + lower)).toLong()
    }
  }
}
