package io.dyte.sockrates.client

import io.dyte.sockrates.client.logger.ExternalLogger
import kotlinx.atomicfu.locks.reentrantLock
import kotlinx.atomicfu.locks.withLock

/** A thread-safe event multicaster to dispatch callbacks for specific events. */
internal class EventManager<EventKeyType, EventListenerType>(
  private val logger: ExternalLogger? = null
) {
  private val listenerMap: MutableMap<EventKeyType, MutableSet<EventListenerType>> = HashMap()

  private val listenersLock = reentrantLock()

  fun addListener(eventKey: EventKeyType, listener: EventListenerType): Boolean {
    return listenersLock.withLock {
      val eventKeyListeners = listenerMap.getOrPut(eventKey) { mutableSetOf() }
      eventKeyListeners.add(listener)
    }
  }

  fun removeListener(eventKey: EventKeyType, listener: EventListenerType): Boolean {
    return listenersLock.withLock { listenerMap[eventKey]?.remove(listener) ?: false }
  }

  fun clearListeners(eventKey: EventKeyType) {
    listenersLock.withLock { listenerMap.remove(eventKey) }
  }

  fun clearAll() {
    listenersLock.withLock { listenerMap.clear() }
  }

  fun multicastEvent(eventKey: EventKeyType, eventBlock: (EventListenerType) -> Unit) {
    val eventKeyListeners = listenersLock.withLock { listenerMap[eventKey]?.toList() } ?: return
    eventKeyListeners.forEach {
      try {
        eventBlock(it)
      } catch (e: Exception) {
        logger?.warn("Sockrates::error in listener of event $eventKey::${e.message}")
      }
    }
  }

  /**
   * Copies the listeners of the `sourceEventKey` if they exist into `destinationEventKey`. NOTE:
   * Currently if destinationEventKey already exists the MultiCaster then its existing listeners are
   * replaced.
   */
  fun copyListeners(sourceEventKey: EventKeyType, destinationEventKey: EventKeyType) {
    listenersLock.withLock {
      val eventKeyListeners = listenerMap[sourceEventKey]
      if (!eventKeyListeners.isNullOrEmpty()) {
        listenerMap[destinationEventKey] = eventKeyListeners.toMutableSet()
      }
    }
  }
}
