@file:OptIn(ExperimentalContracts::class)

package io.dyte.sockrates.client

import io.dyte.sockrates.client.SockratesResult.Failure
import io.dyte.sockrates.client.SockratesResult.Success
import io.ktor.http.Url
import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.InvocationKind
import kotlin.contracts.contract

public sealed class SockratesResult<out SuccessT, out FailureT : FailureReason> {
  data class Success<out SuccessT> internal constructor(
    val value: SuccessT
  ) : SockratesResult<SuccessT, Nothing>()

  data class Failure<out FailureT : FailureReason> internal constructor(
    val reason: FailureT
  ) : SockratesResult<Nothing, FailureT>()
}

public inline fun <SuccessT, FailureT : FailureReason> SockratesResult<SuccessT, FailureT>.onSuccess(
  action: (value: SuccessT) -> Unit
): SockratesResult<SuccessT, FailureT> {
  contract {
    callsInPlace(action, InvocationKind.AT_MOST_ONCE)
  }

  if (this is Success) {
    action(value)
  }

  return this
}

public inline fun <SuccessT, FailureT : FailureReason> SockratesResult<SuccessT, FailureT>.onFailure(
  action: (reason: FailureT) -> Unit
): SockratesResult<SuccessT, FailureT> {
  contract {
    callsInPlace(action, InvocationKind.AT_MOST_ONCE)
  }

  if (this is Failure) {
    action(reason)
  }

  return this
}

open class FailureReason

sealed class ConnectFailureReason(val url: String) : FailureReason() {
  class SocketAlreadyConnected(url: String) : ConnectFailureReason(url)

  class SocketAlreadyConnecting(url: String) : ConnectFailureReason(url)

  class SocketClosing(url: String) : ConnectFailureReason(url)

  class ConnectionError(url: String, val cause: Exception) : ConnectFailureReason(url)

  class InvalidWebSocketUrl(url: String) : ConnectFailureReason(url)
}

sealed class SendFailureReason(
  val event: Int,
  val messageId: String?
) : FailureReason() {
  class SocketNotConnected(
    event: Int,
    messageId: String?
  ) : SendFailureReason(event, messageId)

  class Other(
    event: Int,
    messageId: String?,
    val cause: Exception
  ) : SendFailureReason(event, messageId)
}

sealed class RequestResponseFailureReason(
  val event: Int,
  val messageId: String?
) : FailureReason() {
  class ResponseTimeout(
    event: Int,
    messageId: String?,
    val timeoutInMillis: Long
  ) : RequestResponseFailureReason(event, messageId)

  class SocketNotConnected(
    event: Int,
    messageId: String?
  ) : RequestResponseFailureReason(event, messageId)

  class Other(
    event: Int,
    messageId: String?,
    val cause: Exception
  ) : RequestResponseFailureReason(event, messageId)
}
