package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.models.ParticipantFlags
import io.dyte.core.network.models.UserData
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_SELF_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.UN_MUTE_SELF_AUDIO
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class SelfController(
  private val userData: UserData,
  private val controllerOptions: ControllerOptions,
  controllerContainer: IControllerContainer
) : ISelfController, BaseController(controllerContainer) {
  private lateinit var selfParticipant: DyteSelfParticipant

  override fun init() {
    this.selfParticipant =
      DyteSelfParticipant(
        controllerContainer.metaController.getPeerId(),
        requireNotNull(userData.id),
        controllerOptions.dyteMeetingInfo.displayName,
        userData.picture,
        false,
        userData.clientSpecificId,
        ParticipantFlags(controllerContainer.presetController.canRecord(), false),
        controllerOptions.dyteMeetingInfo.enableAudio,
        controllerOptions.dyteMeetingInfo.enableVideo,
        controllerContainer
      )
  }

  override fun getSelf(): DyteSelfParticipant {
    return selfParticipant
  }

  override fun disableAudio() {
    controllerContainer.platformUtilsProvider.getMediaSoupUtils().muteSelfAudio()
    controllerOptions.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      controllerContainer.socketController.sendMessageSync(MUTE_SELF_AUDIO, null)
    }
  }

  override fun onAudioDisabled() {
    selfParticipant.audioEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override fun enableAudio() {
    if (!controllerContainer.permissionController.isPermissionGrated(MICROPHONE)) {
      throw UnsupportedOperationException("permission not granted to publish audio ")
    }
    if (!controllerContainer.presetController.canPublishAudio()) {
      throw UnsupportedOperationException("not allowed to publish audio ")
    }
    if (!controllerContainer.controllerOptions.dyteMeetingInfo.enableAudio) {
      throw UnsupportedOperationException("not allowed to enableAudio")
    }

    controllerContainer.platformUtilsProvider.getMediaSoupUtils().unmuteSelfAudio()
    runBlocking(Dispatchers.Default) {
      controllerContainer.socketController.sendMessageSync(UN_MUTE_SELF_AUDIO, null)
    }
  }

  override fun onAudioEnabled() {
    selfParticipant.audioEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override fun disableVideo() {
    val producerId =
      controllerContainer.mediaSoupController.videoProduceId
    producerId?.let {
      val content = HashMap<String, JsonElement>()
      content["producerId"] = JsonPrimitive(producerId)
      controllerOptions.platformUtilsProvider.getPlatformUtils().runOnMainThread {
        controllerContainer.socketController.sendMessageSync(MUTE_VIDEO, JsonObject(content))
      }
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().muteSelfVideo()
      controllerOptions.platformUtilsProvider.getPeerConnectionUtils().stopVideo()
    }
  }

  override fun onVideoDisabled() {
    selfParticipant.videoEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun onVideoEnabled(videoProducerId: String) {
    selfParticipant.videoEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun enableVideo() {
    if (!controllerContainer.permissionController.isPermissionGrated(CAMERA)) {
      throw UnsupportedOperationException("permission not granted to publish video")
    }
    if (!controllerContainer.presetController.canPublishVideo()) {
      throw UnsupportedOperationException("not allowed to publish video")
    }
    if (!controllerContainer.controllerOptions.dyteMeetingInfo.enableVideo) {
      throw UnsupportedOperationException("not allowed to enableVideo")
    }

    controllerContainer.mediaSoupController.produceVideo()
    controllerContainer.platformUtilsProvider.getMediaSoupUtils().unmuteSelfVideo()
    controllerOptions.platformUtilsProvider.getPeerConnectionUtils().resumeVideo()
    selfParticipant.videoEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun setDevice(dyteAndroidDevice: DyteAudioDevice) {
    controllerContainer.platformUtilsProvider.getMediaUtils().setAudioDevice(dyteAndroidDevice)
  }

  override fun getAudioDevices(): List<DyteAudioDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getAudioDevices()
  }

  override fun getSelectedAudioDevice(): DyteAudioDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedAudioDevice()
  }

  override fun onAudioDevicesChanged() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnAudioDevicesChanged)
  }

  override fun getVideoDevices(): List<DyteVideoDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getVideoDevices()
  }

  override fun setDevice(device: DyteVideoDevice) {
    controllerContainer.platformUtilsProvider.getMediaUtils().setVideoDevice(device)
  }

  override fun getSelectedVideoDevice(): DyteVideoDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedVideoDevice()
  }

  override fun onVideoDevicesChanged() {
    TODO("Not yet implemented")
  }
}

interface ISelfController {
  fun getSelf(): DyteSelfParticipant

  fun disableAudio()
  fun onAudioDisabled()
  fun enableAudio()
  fun onAudioEnabled()

  fun disableVideo()
  fun onVideoDisabled()
  fun enableVideo()
  fun onVideoEnabled(videoProducerId: String)

  fun getAudioDevices(): List<DyteAudioDevice>
  fun setDevice(dyteAndroidDevice: DyteAudioDevice)
  fun getSelectedAudioDevice(): DyteAudioDevice
  fun onAudioDevicesChanged()

  fun getVideoDevices(): List<DyteVideoDevice>
  fun setDevice(dye: DyteVideoDevice)
  fun getSelectedVideoDevice(): DyteVideoDevice
  fun onVideoDevicesChanged()
}