package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnActiveSpeakerChanged
import io.dyte.core.controllers.DyteEventType.OnAudioDevicesChanged
import io.dyte.core.controllers.DyteEventType.OnCameraClosed
import io.dyte.core.controllers.DyteEventType.OnCameraDisconnected
import io.dyte.core.controllers.DyteEventType.OnCameraError
import io.dyte.core.controllers.DyteEventType.OnCameraFreezed
import io.dyte.core.controllers.DyteEventType.OnCameraOpening
import io.dyte.core.controllers.DyteEventType.OnFirstFrameAvailable
import io.dyte.core.controllers.DyteEventType.OnHostKicked
import io.dyte.core.controllers.DyteEventType.OnMeetingMessagesReceived
import io.dyte.core.controllers.DyteEventType.OnPeerPageUpdate
import io.dyte.core.controllers.DyteEventType.OnMeetingPollsReceived
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingError
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStopped
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomDisconnected
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitCompleted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoined
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeave
import io.dyte.core.controllers.DyteEventType.OnMeetingStateChanged
import io.dyte.core.controllers.DyteEventType.OnNewMeetingMessageReceived
import io.dyte.core.controllers.DyteEventType.OnNewMeetingPollReceived
import io.dyte.core.controllers.DyteEventType.OnNoActiveSpeaker
import io.dyte.core.controllers.DyteEventType.OnPeerAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerJoin
import io.dyte.core.controllers.DyteEventType.OnPeerLeft
import io.dyte.core.controllers.DyteEventType.OnPeerPinned
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerUnpinned
import io.dyte.core.controllers.DyteEventType.OnPeerUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerVideoUpdate
import io.dyte.core.controllers.DyteEventType.OnProximityChanged
import io.dyte.core.controllers.DyteEventType.OnSelfAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfVideoUpdate
import io.dyte.core.listeners.DyteCameraEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DytePermissionEventsListener
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.DyteChatMessage
import io.dyte.core.models.DyteMeetingParticipant
import io.dyte.core.models.DytePlugin
import io.dyte.core.models.DytePollMessage
import kotlinx.serialization.json.JsonObject

internal class EventController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), IEventController {
  override val selfEventsListeners = arrayListOf<DyteSelfEventsListener>()
  override val roomEventsListeners = arrayListOf<DyteMeetingRoomEventsListener>()
  override val participantEventsListeners = arrayListOf<DyteParticipantEventsListener>()
  override val cameraEventsListeners = arrayListOf<DyteCameraEventsListener>()
  override val permissionEventsListeners = arrayListOf<DytePermissionEventsListener>()
  override val pluginEventsListeners = arrayListOf<DytePluginEventsListener>()

  override fun init() {
    // no-op
  }

  override fun triggerEvent(eventType: DyteEventType) {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      triggerEventInternal(eventType)
    }
  }

  private fun triggerEventInternal(eventType: DyteEventType) {
    when (eventType) {
      OnMeetingRoomInitStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitStarted()
        }
      }

      OnMeetingRoomInitCompleted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitCompleted()
        }
      }

      is OnMeetingRoomInitFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitFailed(eventType.exception)
        }
      }

      OnMeetingRoomLeave -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomLeft()
        }
      }

      is OnMeetingStateChanged -> {
        controllerContainer.connectionController.onSocketStateChanged(eventType.state)
      }

      OnMeetingRoomJoinStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomJoinStarted()
        }
      }

      is OnMeetingRoomJoined -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onRoomJoined()
          selfEventsListener.onUpdate(eventType.participant)
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantJoin(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnSelfAudioUpdate -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onAudioUpdate(
            controllerContainer.selfController.getSelf().audioEnabled
          )
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnSelfVideoUpdate -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onVideoUpdate(
            controllerContainer.selfController.getSelf().videoEnabled
          )
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerJoin -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantJoin(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerLeft -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantLeave(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerAudioUpdate -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.audioUpdate(
            eventType.participant.audioEnabled,
            eventType.participant
          )
        }
      }

      is OnPeerVideoUpdate -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.videoUpdate(
            eventType.participant.videoEnabled,
            eventType.participant
          )
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerScreenShareUpdate -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onScreenSharesUpdated()
        }
      }

      is OnPeerUpdate -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantUpdated(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerPageUpdate -> {
        ArrayList(participantEventsListeners).forEach { dyteParticipantEventsListener ->
          dyteParticipantEventsListener.onParticipantsUpdated(
            controllerContainer.participantController.meetingRoomParticipants,
            eventType.isNextPagePossible,
            eventType.isPreviousPagePossible
          )
          dyteParticipantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnActiveSpeakerChanged -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onActiveSpeakerChanged(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnNoActiveSpeaker -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onNoActiveSpeaker()
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerPinned -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantPinned(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnPeerUnpinned -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantUnpinned()
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnMeetingRecordingStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRecordingStarted()
        }
      }

      OnMeetingRecordingStopped -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRecordingEnded()
        }
      }

      is OnMeetingRecordingError -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRecordingStopError(eventType.error)
        }
      }

      is OnMeetingMessagesReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onChatUpdates(false, null, eventType.messages)
        }
      }

      is OnNewMeetingMessageReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onChatUpdates(
            false,
            eventType.message,
            controllerContainer.chatController.dyteChat.messages
          )
        }
      }

      is OnMeetingPollsReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onPollUpdates(false, eventType.polls, null)
        }
      }

      is OnNewMeetingPollReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onPollUpdates(
            false,
            controllerContainer.pollsController.polls,
            eventType.poll
          )
        }
      }

      OnCameraClosed -> {
      }

      OnCameraDisconnected -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          controllerContainer.platformUtilsProvider.getMediaSoupUtils().onCameraStreamKilled()
          controllerContainer.platformUtilsProvider.getPeerConnectionUtils().onCameraClosed()
          cameraEventsListener.onCameraDisconnected()
        }
      }

      is OnCameraError -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          controllerContainer.platformUtilsProvider.getMediaSoupUtils().onCameraStreamKilled()
          controllerContainer.platformUtilsProvider.getPeerConnectionUtils().onCameraClosed()
          cameraEventsListener.onCameraError(eventType.message)
        }
      }

      is OnCameraFreezed -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onCameraFreezed(eventType.message)
        }
      }

      is OnCameraOpening -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onCameraOpening(eventType.message)
        }
      }

      OnFirstFrameAvailable -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onFirstFrameAvailable()
        }
      }

      OnAudioDevicesChanged -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onAudioDevicesUpdated()
        }
      }

      is OnProximityChanged -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onProximityChanged(eventType.isNear)
        }
      }

      OnMeetingRoomDisconnected -> {
        ArrayList(roomEventsListeners).forEach { roomEventListener ->
          roomEventListener.onMeetingRoomDisconnected()
        }
      }

      is DyteEventType.OnPluginEnabled -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginActivated(eventType.plugin)
        }
      }

      is DyteEventType.OnPluginDisabled -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginDeactivated(eventType.plugin)
        }
      }

      is DyteEventType.OnPluginMessage -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginMessage(eventType.message)
        }
      }

      OnHostKicked -> {
        ArrayList(roomEventsListeners).forEach { roomEventListener ->
          roomEventListener.onHostKicked()
        }
      }
    }
  }

  override fun addSelfEventListener(selfEventListener: DyteSelfEventsListener) {
    this.selfEventsListeners.add(selfEventListener)
  }

  override fun removeSelfEventListener(selfEventListener: DyteSelfEventsListener) {
    if (this.selfEventsListeners.contains(selfEventListener)) {
      this.selfEventsListeners.remove(selfEventListener)
    }
  }

  override fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    this.roomEventsListeners.add(roomEventsListener)
  }

  override fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    if (this.roomEventsListeners.contains(roomEventsListener)) {
      this.roomEventsListeners.remove(roomEventsListener)
    }
  }

  override fun addParticipantEventListener(participantEventsListener: DyteParticipantEventsListener) {
    this.participantEventsListeners.add(participantEventsListener)
  }

  override fun removeParticipantEventListener(participantEventsListener: DyteParticipantEventsListener) {
    if (this.participantEventsListeners.contains(participantEventsListener)) {
      this.participantEventsListeners.remove(participantEventsListener)
    }
  }

  override fun addCameraEventListener(cameraEventsListener: DyteCameraEventsListener) {
    this.cameraEventsListeners.add(cameraEventsListener)
  }

  override fun removeCameraEventListener(cameraEventsListener: DyteCameraEventsListener) {
    if (this.cameraEventsListeners.contains(cameraEventsListener)) {
      this.cameraEventsListeners.remove(cameraEventsListener)
    }
  }

  override fun addPermissionEventListener(permissionEventsListener: DytePermissionEventsListener) {
    this.permissionEventsListeners.add(permissionEventsListener)
  }

  override fun removePermissionEventListener(permissionEventsListener: DytePermissionEventsListener) {
    if (this.permissionEventsListeners.contains(permissionEventsListener)) {
      this.permissionEventsListeners.remove(permissionEventsListener)
    }
  }

  override fun addPluginEventListener(pluginEventsListener: DytePluginEventsListener) {
    this.pluginEventsListeners.add(pluginEventsListener)
  }

  override fun removePluginEventListener(pluginEventsListener: DytePluginEventsListener) {
    if (this.pluginEventsListeners.contains(pluginEventsListener)) {
      this.pluginEventsListeners.remove(pluginEventsListener)
    }
  }

  override fun dispose() {
    this.selfEventsListeners.clear()
    this.roomEventsListeners.clear()
    this.participantEventsListeners.clear()
    this.cameraEventsListeners.clear()
    this.pluginEventsListeners.clear()
  }
}

interface IEventController {
  val selfEventsListeners: List<DyteSelfEventsListener>
  val roomEventsListeners: List<DyteMeetingRoomEventsListener>
  val participantEventsListeners: List<DyteParticipantEventsListener>
  val cameraEventsListeners: List<DyteCameraEventsListener>
  val permissionEventsListeners: List<DytePermissionEventsListener>
  val pluginEventsListeners: List<DytePluginEventsListener>

  fun addSelfEventListener(selfEventListener: DyteSelfEventsListener)
  fun removeSelfEventListener(selfEventListener: DyteSelfEventsListener)

  fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)
  fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)

  fun addParticipantEventListener(participantEventsListener: DyteParticipantEventsListener)
  fun removeParticipantEventListener(participantEventsListener: DyteParticipantEventsListener)

  fun addCameraEventListener(cameraEventsListener: DyteCameraEventsListener)
  fun removeCameraEventListener(cameraEventsListener: DyteCameraEventsListener)

  fun addPermissionEventListener(permissionEventsListener: DytePermissionEventsListener)
  fun removePermissionEventListener(permissionEventsListener: DytePermissionEventsListener)

  fun addPluginEventListener(pluginEventsListener: DytePluginEventsListener)
  fun removePluginEventListener(pluginEventsListener: DytePluginEventsListener)

  fun dispose()

  fun triggerEvent(eventType: DyteEventType)
}

sealed class DyteEventType {
  object OnMeetingRoomInitStarted : DyteEventType()
  object OnMeetingRoomInitCompleted : DyteEventType()
  class OnMeetingRoomInitFailed(val exception: Exception) : DyteEventType()

  object OnMeetingRoomLeave : DyteEventType()
  object OnMeetingRoomLeaveStarted : DyteEventType()
  object OnMeetingRoomDisconnected : DyteEventType()

  object OnMeetingRoomJoinStarted : DyteEventType()
  class OnMeetingRoomJoined(
    val participant: DyteMeetingParticipant
  ) :
    DyteEventType()

  class OnMeetingStateChanged(val state: String, val isReconnected: Boolean = false) :
    DyteEventType()

  // participant related event callbacks
  class OnPeerJoin(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerLeft(val participant: DyteMeetingParticipant) : DyteEventType()
  object OnPeerScreenShareUpdate : DyteEventType()
  class OnPeerAudioUpdate(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerVideoUpdate(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerPinned(val participant: DyteMeetingParticipant) : DyteEventType()
  object OnPeerUnpinned : DyteEventType()
  class OnPeerUpdate(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerPageUpdate(val isNextPagePossible: Boolean, val isPreviousPagePossible: Boolean) :
    DyteEventType()

  object OnSelfAudioUpdate : DyteEventType()
  object OnSelfVideoUpdate : DyteEventType()

  class OnActiveSpeakerChanged(val participant: DyteMeetingParticipant) : DyteEventType()
  object OnNoActiveSpeaker : DyteEventType()

  object OnAudioDevicesChanged : DyteEventType()
  class OnProximityChanged(val isNear: Boolean) : DyteEventType()

  object OnMeetingRecordingStarted : DyteEventType()
  object OnMeetingRecordingStopped : DyteEventType()
  class OnMeetingRecordingError(val error: Exception) : DyteEventType()

  class OnMeetingPollsReceived(val polls: List<DytePollMessage>) : DyteEventType()
  class OnNewMeetingPollReceived(val poll: DytePollMessage) : DyteEventType()

  class OnMeetingMessagesReceived(val messages: List<DyteChatMessage>) : DyteEventType()
  class OnNewMeetingMessageReceived(val message: DyteChatMessage) : DyteEventType()

  class OnPermissionsUpdated(val permissions: List<Permission>) : DyteEventType()

  class OnCameraError(val message: String?) : DyteEventType()
  object OnCameraDisconnected : DyteEventType()
  class OnCameraFreezed(val message: String?) : DyteEventType()
  class OnCameraOpening(val message: String?) : DyteEventType()
  object OnFirstFrameAvailable : DyteEventType()
  object OnCameraClosed : DyteEventType()

  object OnHostKicked: DyteEventType()

  class OnPluginEnabled(val plugin: DytePlugin): DyteEventType()
  class OnPluginDisabled(val plugin: DytePlugin): DyteEventType()
  class OnPluginMessage(val message: JsonObject): DyteEventType()
}