package io.dyte.core.media

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import androidx.cardview.widget.CardView
import io.dyte.core.DyteMobileClient
import io.dyte.core.R
import io.dyte.core.Utils
import io.dyte.core.models.DyteMeetingParticipant
import io.dyte.core.models.DyteSelfParticipant
import org.webrtc.EglBase
import org.webrtc.SurfaceViewRenderer
import org.webrtc.VideoTrack

class VideoView : FrameLayout, VideoRenderer {
  private lateinit var videoRenderer: SurfaceViewRenderer
  private lateinit var ivCamera: ImageView
  private lateinit var ivMic: ImageView
  private lateinit var tvName: TextView
  private lateinit var tvInitials: TextView
  private lateinit var cvVideoContainer: CardView

  private var videoTrack: VideoTrack? = null

  private var isEglContextAttached = false

  constructor(context: Context) : super(context) {
    init(context)
  }

  constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
    init(context)
  }

  constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
    context,
    attrs,
    defStyleAttr
  ) {
    init(context)
  }

  private fun init(context: Context) {
    val view = LayoutInflater.from(context).inflate(R.layout.view_video, this, false)
    addView(view)

    videoRenderer = view.findViewById(R.id.video_renderer)

    ivMic = view.findViewById(R.id.ivMic)
    ivCamera = view.findViewById(R.id.ivVideo)
    tvName = view.findViewById(R.id.tvName)
    tvInitials = view.findViewById(R.id.tvSelfInitials)
    cvVideoContainer = view.findViewById(R.id.cvVideoContainer)
  }

  override fun render(
    dyteParticipant: DyteMeetingParticipant,
    client: DyteMobileClient,
    onTop: Boolean,
    isScreenSharePeer: Boolean
  ) {
    if (isScreenSharePeer) {
      cvVideoContainer.visibility = VISIBLE
      val track = dyteParticipant.screenShareTrack as VideoTrack
      if (!isEglContextAttached) {
        videoRenderer.init(client.localUser.getEglContext() as EglBase.Context, null)
        isEglContextAttached = true
      }
      track.addSink(videoRenderer)
    } else {
      if (dyteParticipant.videoEnabled && dyteParticipant.videoTrack is VideoTrack) {
        val track = dyteParticipant.videoTrack as VideoTrack
        if (!isEglContextAttached) {
          videoRenderer.init(client.localUser.getEglContext() as EglBase.Context, null)
          isEglContextAttached = true
        }
        cvVideoContainer.visibility = VISIBLE
        videoRenderer.visibility = VISIBLE
        if (onTop || dyteParticipant is DyteSelfParticipant) {
          videoRenderer.setZOrderMediaOverlay(true)
        } else {
          videoRenderer.setZOrderMediaOverlay(false)
        }
        videoTrack = track
        track.addSink(videoRenderer)
      } else {
        cvVideoContainer.visibility = INVISIBLE
        videoRenderer.visibility = INVISIBLE
      }
    }
    refresh(dyteParticipant, isScreenSharePeer)
  }

  private fun refresh(participant: DyteMeetingParticipant, isScreenSharePeer: Boolean) {
    if (participant.audioEnabled) {
      ivMic.setImageResource(R.drawable.ic_baseline_mic_24)
    } else {
      ivMic.setImageResource(R.drawable.ic_baseline_mic_off_24)
    }

    if (participant.videoEnabled || isScreenSharePeer) {
      ivCamera.setImageResource(R.drawable.ic_baseline_videocam_24)
      tvInitials.visibility = View.GONE
    } else {
      tvInitials.visibility = View.VISIBLE
      ivCamera.setImageResource(R.drawable.ic_baseline_videocam_off_24)
    }

    if (isScreenSharePeer) {
      tvName.text = participant.name + "'s Screen"
      ivMic.visibility = View.GONE
      ivCamera.visibility = View.GONE
    } else {
      tvName.text = participant.name
    }

    tvInitials.text = Utils.getInitialsFromName(participant.name)
  }

  fun release() {
    videoRenderer.release()
    // videoTrack?.removeSink(videoRenderer)
    if (isEglContextAttached) {
      isEglContextAttached = false
    }
  }
}