package io.dyte.core.network

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.network.models.GraphQlRequest
import io.dyte.core.network.models.GraphQlRequestVariables
import io.dyte.core.network.models.IceServersWrapper
import io.dyte.core.network.models.MeetingSessionDataWrapper
import io.dyte.core.network.models.MultiplePluginResponse
import io.dyte.core.network.models.PluginAuthResponse
import io.dyte.core.network.models.PluginAuthorizationModel
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.network.models.PluginResponse
import io.dyte.core.network.models.RecordingData
import io.dyte.core.network.models.RecordingResponseWrapper
import io.dyte.core.network.models.StopRecordingModel
import io.dyte.core.network.models.UserDataWrapper
import io.dyte.core.network.models.UserPresetDataWrapper
import io.dyte.core.network.models.UserPresetRequestModel
import io.dyte.core.observability.PostDyteLogs
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.logging.LogLevel.ALL
import io.ktor.client.plugins.logging.LogLevel.NONE
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.plugins.logging.SIMPLE
import io.ktor.client.request.get
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.client.statement.*
import io.ktor.http.ContentType
import io.ktor.http.contentType
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json

private const val VERSION = "0.5.0"
private const val CLIENT_TYPE = "CLIENT_APP"
private const val GRAPHQL_REQUEST_QUERY: String =
  "\nquery Session(" + "$" + "roomName: String!, " + "$" + "password: String) {\n\tsession(roomName: " + "$" + "roomName, password: " + "$" + "password) {\n\t  title,\n\t  roomNodeLink,\n\t  roomName,\n\t  password\n\t}\n  }\n"

internal class ApiClient(
  private val controllerContainer: IControllerContainer
) : IApiClient {
  internal val client: HttpClient = HttpClient {
    install(Logging) {
      logger = Logger.SIMPLE
      level = if (controllerContainer.platformUtilsProvider.getLogger().isLoggerEnabled()) ALL else NONE
    }
    install(ContentNegotiation) {
      json(Json {
        prettyPrint = true
        isLenient = true
        ignoreUnknownKeys = true
      })
    }
  }

  val json = Json {
    ignoreUnknownKeys = true
  }

  override suspend fun getRoomNodeData(): MeetingSessionDataWrapper {
    val roomName = controllerContainer.metaController.getRoomName()
    val graphQlRequestVariables = GraphQlRequestVariables(roomName, "")
    val requestData = GraphQlRequest(GRAPHQL_REQUEST_QUERY, graphQlRequestVariables)
    val response = client.post("${controllerContainer.metaController.getBaseUrl()}/graphql") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      setBody(requestData)
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun getICEServers(): IceServersWrapper {
    val endpoint = "${controllerContainer.metaController.getBaseUrl()}/iceservers"
    val response = client.get(endpoint) {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
    }
    return response.body()
  }

  override suspend fun getUserDetails(): UserDataWrapper {
    val response =
      client.get("${controllerContainer.metaController.getBaseUrl()}/auth/basicUserDetails?authToken=${controllerContainer.metaController.getAuthToken()}") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      }
    return response.body()
  }

  override suspend fun getUserPreset(): UserPresetDataWrapper {
    val roomName = controllerContainer.metaController.getRoomName()
    val response = client.post("${controllerContainer.metaController.getBaseUrl()}/v1/userpreset") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      setBody(
        UserPresetRequestModel(
          controllerContainer.metaController.getAuthToken(),
          CLIENT_TYPE,
          roomName,
          VERSION
        )
      )
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun startRecording(): RecordingResponseWrapper {
    val response =
      client.post("${controllerContainer.metaController.getBaseUrl()}/v1/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/recording") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        contentType(ContentType.Application.Json)
      }
    return response.body()
  }

  override suspend fun stopRecording(recordingData: RecordingData) {
      client.put("${controllerContainer.metaController.getBaseUrl()}/v1/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/recordings/${recordingData.id}") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        setBody(
          StopRecordingModel("stop")
        )
        contentType(ContentType.Application.Json)
      }
  }

  override suspend fun getActiveRecording(): RecordingResponseWrapper {
    val response = client.get("${controllerContainer.metaController.getBaseUrl()}/v1/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/active-recording/") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun getPlugins(): MultiplePluginResponse {
    val response = client.get("${controllerContainer.metaController.getBaseUrl()}/v2/plugins/user") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun getPluginDetails(pluginId: String): PluginResponse {
    val response = client.get("${controllerContainer.metaController.getBaseUrl()}/v2/plugins/view/$pluginId") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse {
    val response = client.get("${pluginBaseUrl}/dyte-config.json") {
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun authorizePlugin(pluginId: String): PluginAuthResponse {

    val response = client.post("${controllerContainer.metaController.getBaseUrl()}/v2/plugins/authorize/$pluginId") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
      setBody(
        PluginAuthorizationModel(
        controllerContainer.metaController.getRoomName(),
        controllerContainer.metaController.getPeerId()
      )
      )
    }
    return response.body()
  }

  override suspend fun uploadLogs(logs: PostDyteLogs) {
    client.post("https://api-silos.dyte.io/otel/logs") {
      contentType(ContentType.Application.Json)
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      setBody(logs)
    }
  }

  override fun getClient(): HttpClient {
    return client
  }
}

interface IApiClient {
  suspend fun getRoomNodeData(): MeetingSessionDataWrapper
  suspend fun getUserPreset(): UserPresetDataWrapper
  suspend fun getUserDetails(): UserDataWrapper
  suspend fun getICEServers(): IceServersWrapper
  suspend fun startRecording(): RecordingResponseWrapper
  suspend fun stopRecording(recordingData: RecordingData)
  suspend fun getActiveRecording(): RecordingResponseWrapper
  suspend fun getPlugins(): MultiplePluginResponse
  suspend fun getPluginDetails(pluginId: String): PluginResponse
  suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse
  suspend fun authorizePlugin(pluginId: String): PluginAuthResponse
  suspend fun uploadLogs(logs: PostDyteLogs)
  fun getClient(): HttpClient
}