package io.dyte.core.controllers

import com.dyte.mobilecorekmm.platform.IDyteMediaSoupClientCallback
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.listeners.DyteCameraEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.DyteChat
import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeta
import io.dyte.core.models.DytePlugins
import io.dyte.core.models.DytePoll
import io.dyte.core.models.DyteRecording
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.ApiClient
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.IApiClient
import io.dyte.core.network.models.UserData
import io.dyte.core.observability.DyteLoggerController
import io.dyte.core.observability.IDyteLoggerController
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.SocketMessageResponseParser
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerConnectModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

/**
 *
 */
internal class Controller : IController, IControllerContainer, IDyteMediaSoupClientCallback {

  private lateinit var _chatController: ChatController
  override val chatController: IChatController
    get() = _chatController

  private lateinit var _participantController: ParticipantController
  override val participantController: IParticipantController
    get() = this._participantController

  private lateinit var _pollController: PollController
  override val pollsController: IPollController
    get() = _pollController

  private lateinit var _metaController: MetaController
  override val metaController: IMetaController
    get() = _metaController

  private lateinit var _selfController: SelfController
  override val selfController: ISelfController
    get() = this._selfController

  private lateinit var _roomNodeController: RoomNodeController
  override val roomNodeController: IRoomNodeController
    get() = this._roomNodeController

  private lateinit var _presetController: PresetController
  override val presetController: IPresetController
    get() = this._presetController

  private lateinit var _hostController: HostController
  override val hostController: IHostController
    get() = this._hostController

  private lateinit var _apiClient: IApiClient
  override val apiClient: IApiClient
    get() = this._apiClient

  private lateinit var _socketController: SocketController
  override val socketController: ISocketController
    get() = this._socketController

  private var _eventController = EventController(this)
  override val eventController: IEventController
    get() = this._eventController

  private var _connectionController = ConnectionController(this)
  override val connectionController: IConnectionController
    get() = this._connectionController

  private lateinit var _socketMessageResponseParser: SocketMessageResponseParser
  override val socketMessageResponseParser: ISocketMessageResponseParser
    get() = this._socketMessageResponseParser

  private lateinit var _platformUtilsProvider: IDytePlatformUtilsProvider
  override val platformUtilsProvider: IDytePlatformUtilsProvider
    get() = this._platformUtilsProvider

  private lateinit var _recordingController: RecordingController
  override val recordingController: IRecordingController
    get() = this._recordingController

  private lateinit var _permissionController: PermissionController
  override val permissionController: IPermissionController
    get() = this._permissionController

  private lateinit var _pluginsController: PluginsController
  override val pluginsController: IPluginsController
    get() = this._pluginsController

  private lateinit var _controllerOptions: ControllerOptions
  override val controllerOptions: ControllerOptions
    get() = this._controllerOptions

  private lateinit var _loggerController: DyteLoggerController
  override val loggerController: IDyteLoggerController
    get() = this._loggerController

  private lateinit var userData: UserData

  private lateinit var recording: DyteRecording
  private lateinit var poll: DytePoll
  private lateinit var meta: DyteMeta
  private lateinit var plugins: DytePlugins

  private val jsonParser = BaseApiService.json

  init {
    this._eventController.init()
  }

  override fun init(controllerOptions: ControllerOptions) {
    this._controllerOptions = controllerOptions
    this._platformUtilsProvider = controllerOptions.platformUtilsProvider
    _platformUtilsProvider.getPlatformUtils().listenForCrashes()
    this.eventController.triggerEvent(DyteEventType.OnMeetingRoomInitStarted)
    platformUtilsProvider.init(this@Controller, this@Controller)

    _metaController = MetaController(controllerOptions.dyteMeetingInfo, this@Controller)
    _metaController.init()

    _permissionController = PermissionController(this)
    _permissionController.init()

    _loggerController = DyteLoggerController(this@Controller)
    _loggerController.init()

    val permissionsToAsk = arrayListOf<PermissionType>()
    if (controllerOptions.dyteMeetingInfo.enableVideo && _permissionController.isPermissionGrated(CAMERA).not()) {
      permissionsToAsk.add(CAMERA)
    }
    if (controllerOptions.dyteMeetingInfo.enableAudio && _permissionController.isPermissionGrated(MICROPHONE).not()) {
      permissionsToAsk.add(MICROPHONE)
    }

    _permissionController.askPermissions(permissionsToAsk,
      {
        loggerController.traceLog("permissions granted")
        doInit(controllerOptions)
      },
      {
        loggerController.traceLog("permissions not granted")
        doInit(controllerOptions)
      })
  }

  private fun doInit(controllerOptions: ControllerOptions) {
    _platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        platformUtilsProvider.getLogger().enableLogger(true)

        _connectionController = ConnectionController(this@Controller)
        _apiClient = ApiClient(this@Controller)

        try {
          userData = runBlocking(Dispatchers.Default) {
            getUsersData()
          }
        } catch (e: Exception) {
          e.printStackTrace()
          eventController.triggerEvent(DyteEventType.OnMeetingRoomInitFailed(e))
          loggerController.traceError("failed to fetch user data ${e.message}")
          return@runBlocking
        }

        _presetController = PresetController(this@Controller)
        _socketController = SocketController(this@Controller)
        _participantController = ParticipantController(this@Controller)
        _selfController = SelfController(userData, controllerOptions, this@Controller)
        _chatController = ChatController(this@Controller)
        _pollController = PollController(this@Controller)
        _hostController = HostController(this@Controller)
        _recordingController = RecordingController(this@Controller)
        _pluginsController = PluginsController(this@Controller)
        _pluginsController.init()

        recording = DyteRecording(this@Controller)
        poll = DytePoll(this@Controller)
        meta = DyteMeta(this@Controller)
        plugins = DytePlugins(this@Controller)

        _permissionController.init()
        _connectionController.init()
        _socketController.init()
        _presetController.init()
        _participantController.init()
        _selfController.init()
        _chatController.init()
        _pollController.init()
        _hostController.init()
        _recordingController.init()

        withContext(Dispatchers.Main) {
          platformUtilsProvider.getMediaSoupUtils().init(this@Controller)
          platformUtilsProvider.getMediaUtils().init()
        }

        metaController.setMeetingTitle(controllerOptions.dyteMeetingInfo.roomName)

        _socketMessageResponseParser = SocketMessageResponseParser()
        _roomNodeController = RoomNodeController(this@Controller)
        _roomNodeController.init()

        socketController.connect()
        // eventController.triggerEvent(DyteEventType.OnMeetingRoomInitCompleted)
      }
    }
  }

  override fun joinRoom() {
    platformUtilsProvider.getPlatformUtils().runOnIoThread {
      roomNodeController.joinRoom()
    }
  }

  override fun leaveRoom() {
    eventController.triggerEvent(DyteEventType.OnMeetingRoomLeaveStarted)
    platformUtilsProvider.getMediaSoupUtils().leaveCall()
    socketController.disconnect()
    eventController.triggerEvent(DyteEventType.OnMeetingRoomLeave)
  }

  override fun getSelf(): DyteSelfParticipant {
    return this.selfController.getSelf()
  }

  override fun onReceiveTransportConnected(transportId: String, dtlsParameters: String) {
    loggerController.traceLog("onReceiveTransportConnected")
    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(transportId)
    content["dtlsParameters"] = jsonParser.parseToJsonElement(dtlsParameters)
    val payload = JsonObject(content)
    this.socketController.sendMessageSync(
      OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
      payload
    )
  }

  override fun onSendTransportConnected(transportId: String, dtlsParameters: String) {
    loggerController.traceLog("onSendTransportConnected")
    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(transportId)
    content["dtlsParameters"] = jsonParser.parseToJsonElement(dtlsParameters)
    val payload = JsonObject(content)
    this.socketController.sendMessageSync(
      OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
      payload
    )
  }

  override fun onProduce(
    transportId: String,
    kind: String,
    rtpParameters: String,
    appData: String?
  ): String {
    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(transportId)
    rtpParameters.let { content["rtpParameters"] = jsonParser.parseToJsonElement(it) }
    content["kind"] = JsonPrimitive(kind)
    appData?.let { content["appData"] = jsonParser.parseToJsonElement(it) }

    val produceResponse =
      this.socketController.sendMessageSync(OutboundMeetingEventType.PRODUCE, JsonObject(content))
    val producerConnectModel =
      this.socketMessageResponseParser.parseResponse(produceResponse).payload as WebSocketProducerConnectModel
    if (kind == "video") {
      this.platformUtilsProvider.getMediaSoupUtils()
        .setVideoProducerId(requireNotNull(producerConnectModel.id))
    }
    return requireNotNull(producerConnectModel.id)
  }

  override fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    this.eventController.addRoomEventListener(meetingRoomEventsListener)
  }

  override fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    this.eventController.removeRoomEventListener(meetingRoomEventsListener)
  }

  override fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    this.eventController.addSelfEventListener(selfEventsListener)
  }

  override fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    this.eventController.removeSelfEventListener(selfEventsListener)
  }

  override fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    this.eventController.addParticipantEventListener(participantEventsListener)
  }

  override fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    this.eventController.removeParticipantEventListener(participantEventsListener)
  }

  override fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    this.eventController.addCameraEventListener(cameraEventsListener)
  }

  override fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    this.eventController.removeCameraEventListener(cameraEventsListener)
  }

  private suspend fun getUsersData(): UserData {
    return requireNotNull(this.apiClient.getUserDetails().user)
  }

  override fun getMeetingParticipant(): DyteRoomParticipants {
    return this.participantController.meetingRoomParticipants
  }

  override fun getChat(): DyteChat {
    return chatController.dyteChat
  }

  override fun getRecording(): DyteRecording {
    return recording
  }

  override fun getPolls(): DytePoll {
    return poll
  }

  override fun getMeta(): DyteMeta {
    return meta
  }

  override fun getPlugins(): DytePlugins {
    return plugins
  }
}

internal interface IController {
  fun init(controllerOptions: ControllerOptions)
  fun joinRoom()
  fun leaveRoom()

  fun getSelf(): DyteSelfParticipant
  fun getRecording(): DyteRecording

  fun getMeetingParticipant(): DyteRoomParticipants
  fun getChat(): DyteChat
  fun getPolls(): DytePoll
  fun getMeta(): DyteMeta
  fun getPlugins(): DytePlugins

  /**
   * Add meeting room events listener
   *
   * @param meetingRoomEventsListener
   */
  fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)
  fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)

  /**
   * Add self events listener
   *
   * @param selfEventsListener
   */
  fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener)
  fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener)

  /**
   * Add participant events listener
   *
   * @param participantEventsListener
   */
  fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)
  fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)

  fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)
  fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)
}

interface IControllerContainer {
  val controllerOptions: ControllerOptions
  val chatController: IChatController
  val participantController: IParticipantController
  val pollsController: IPollController
  val metaController: IMetaController
  val selfController: ISelfController
  val roomNodeController: IRoomNodeController
  val presetController: IPresetController
  val hostController: IHostController
  val apiClient: IApiClient
  val socketController: ISocketController
  val socketMessageResponseParser: ISocketMessageResponseParser
  val platformUtilsProvider: IDytePlatformUtilsProvider
  val recordingController: IRecordingController
  val eventController: IEventController
  val connectionController: IConnectionController
  val permissionController: IPermissionController
  val pluginsController: IPluginsController
  val loggerController: IDyteLoggerController
}

data class ControllerOptions(
  val dyteMeetingInfo: DyteMeetingInfo,
  val platformUtilsProvider: IDytePlatformUtilsProvider
)
