package io.dyte.core.platform

import android.annotation.SuppressLint
import android.util.Log
import android.webkit.JavascriptInterface
import android.webkit.WebViewClient
import io.dyte.core.platform.IDyteWebView.IDyteWebViewListener
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonObject

actual typealias WebView = android.webkit.WebView

@SuppressLint("SetJavaScriptEnabled")
class DyteAndroidWebView(private val utilsProvider: IDytePlatformUtilsProvider) : IDyteWebView {
  private var webView: WebView? = null
  private var url: String? = null
  private var listener: IDyteWebViewListener? = null

  override fun setURL(url: String) {
    this.url = url
    webView?.loadUrl(url)
  }

  override fun unbindWebView() {
    webView = null
  }

  override fun setListener(listener: IDyteWebViewListener) {
    this.listener = listener
  }

  override fun removeListener() {
    this.listener = null
  }

  override fun bindWebView(webView: WebView) {
    with(webView) {
      settings.javaScriptEnabled = true
      settings.domStorageEnabled = true
      settings.databaseEnabled = true

      clearCache(true)
      clearHistory()

      webView.addJavascriptInterface(WebViewInterface(listener!!), NAME_JS_INTERFACE)
      webView.webViewClient = object : WebViewClient() {
        override fun onPageFinished(view: android.webkit.WebView?, url: String?) {
          super.onPageFinished(view, url)
          view?.evaluateJavascript(jsListenerScript) {
            Log.d("PluginFeat", "DyteAndroidWebView.onPageFinished Done!")
          }
        }
      }
    }
    this.webView = webView
    if (url != null) {
      this.webView?.loadUrl(url as String)
    } else {
      println("DyteAndroidWebView => bindWebView url is null")
    }
  }

  override fun submitEvent(payload: JsonObject) {
    webView?.evaluateJavascript(
      """
        window.postMessage($payload);
      """.trimIndent(),
      null
    )
  }

  companion object {
    private const val NAME_JS_INTERFACE = "WebViewInterface"
    private val jsListenerScript = """
      window.parent = {};
      window.parent.postMessage = function(message) {
        var formattedJSON = JSON.stringify(message);
        $NAME_JS_INTERFACE.postMessage(formattedJSON);
      };
      """.trimIndent()
  }

  private class WebViewInterface(private val webViewListener: IDyteWebViewListener) {
    @JavascriptInterface
    fun postMessage(messageJson: String) {
      webViewListener.onWebViewMessage(Json.parseToJsonElement(messageJson).jsonObject)
    }
  }
}