package io.dyte.core.platform

import android.app.Application
import android.content.Context
import android.text.TextUtils
import io.dyte.core.controllers.DyteEventType
import org.webrtc.AudioSource
import org.webrtc.Camera1Enumerator
import org.webrtc.Camera2Enumerator
import org.webrtc.CameraEnumerator
import org.webrtc.CameraVideoCapturer
import org.webrtc.CameraVideoCapturer.CameraEventsHandler
import org.webrtc.DefaultVideoDecoderFactory
import org.webrtc.DefaultVideoEncoderFactory
import org.webrtc.EglBase
import org.webrtc.MediaConstraints
import org.webrtc.PeerConnectionFactory
import org.webrtc.SurfaceTextureHelper
import org.webrtc.VideoDecoderFactory
import org.webrtc.VideoEncoderFactory
import org.webrtc.VideoSource
import org.webrtc.audio.AudioDeviceModule
import org.webrtc.audio.JavaAudioDeviceModule
import org.webrtc.audio.JavaAudioDeviceModule.AudioRecordErrorCallback
import org.webrtc.audio.JavaAudioDeviceModule.AudioTrackErrorCallback

internal class DyteAndroidPeerConnection(private val utilsProvider: IDytePlatformUtilsProvider) :
  IDytePeerConnectionUtils {
  private var mPeerConnectionFactory: PeerConnectionFactory? = null
  private var mAudioSource: AudioSource? = null
  private var mVideoSource: VideoSource? = null
  private var mCamCapture: CameraVideoCapturer? = null

  private var eglBaseContext: EglBase.Context = EglBase.create().eglBaseContext

  override fun switchCamera() {
    requireNotNull(mCamCapture).switchCamera(null)
  }

  override fun createAudioTrack(id: String): Any {
    if (mAudioSource == null) {
      createAudioSource(
        utilsProvider.getPlatformUtils().getAndroidApplicationContext() as Application
      )
    }
    return requireNotNull(mPeerConnectionFactory).createAudioTrack(id, mAudioSource)
  }

  override fun createVideoTrack(id: String): Any {
    if (mVideoSource == null) {
      createVideoSource(
        utilsProvider.getPlatformUtils().getAndroidApplicationContext() as Application
      )
    }
    return requireNotNull(mPeerConnectionFactory).createVideoTrack(id, mVideoSource)
  }

  override fun getEglContext(): EglBase.Context {
    return eglBaseContext
  }

  override fun dispose() {
    if (mCamCapture != null) {
      try {
        mCamCapture?.stopCapture()
        mCamCapture?.dispose()
        mCamCapture = null
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    if (mVideoSource != null) {
      try {
        mVideoSource?.dispose()
        mVideoSource = null
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    if (mAudioSource != null) {
      try {
        mAudioSource?.dispose()
        mAudioSource = null
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    if (mPeerConnectionFactory != null) {
      mPeerConnectionFactory?.dispose()
      mPeerConnectionFactory = null
    }

  }

  override fun onCameraClosed() {
    mCamCapture = null
    mVideoSource = null
  }

  override fun stopVideo() {
    mCamCapture?.stopCapture()
  }

  override fun resumeVideo() {
    mCamCapture?.startCapture(640, 480, 30)
  }

  private fun createPeerConnectionFactory(context: Context) {
    utilsProvider.getControllerContainer().loggerController.traceLog("create peer connection factory")
    val builder = PeerConnectionFactory.builder()
    builder.setOptions(null)
    val adm = createJavaAudioDevice(context)
    val encoderFactory: VideoEncoderFactory = DefaultVideoEncoderFactory(
      getEglContext(), true /* enableIntelVp8Encoder */, true
    )
    val decoderFactory: VideoDecoderFactory =
      DefaultVideoDecoderFactory(getEglContext())

    val peerConnectionFactoryOpptionsBuilder =
      PeerConnectionFactory.InitializationOptions
        .builder(utilsProvider.getPlatformUtils().getAndroidApplicationContext() as Application)
        .setNativeLibraryName("mediasoupclient_so")
        .createInitializationOptions()

    PeerConnectionFactory.initialize(peerConnectionFactoryOpptionsBuilder)

    mPeerConnectionFactory = builder
      .setAudioDeviceModule(adm)
      .setVideoEncoderFactory(encoderFactory)
      .setVideoDecoderFactory(decoderFactory)
      .createPeerConnectionFactory()
  }

  private fun createJavaAudioDevice(appContext: Context): AudioDeviceModule {
    utilsProvider.getControllerContainer().loggerController.traceLog("create java audio device")
    val audioRecordErrorCallback: AudioRecordErrorCallback = object : AudioRecordErrorCallback {
      override fun onWebRtcAudioRecordInitError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio record init error $errorMessage")
      }

      override fun onWebRtcAudioRecordStartError(
        errorCode: JavaAudioDeviceModule.AudioRecordStartErrorCode,
        errorMessage: String
      ) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio record start error $errorMessage")
      }

      override fun onWebRtcAudioRecordError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio record error $errorMessage")
      }
    }
    val audioTrackErrorCallback: AudioTrackErrorCallback = object : AudioTrackErrorCallback {
      override fun onWebRtcAudioTrackInitError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio track init error $errorMessage")
      }

      override fun onWebRtcAudioTrackStartError(
        errorCode: JavaAudioDeviceModule.AudioTrackStartErrorCode,
        errorMessage: String
      ) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio track start error $errorMessage")
      }

      override fun onWebRtcAudioTrackError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio track error $errorMessage")
      }
    }
    return JavaAudioDeviceModule.builder(appContext)
      .setAudioRecordErrorCallback(audioRecordErrorCallback)
      .setAudioTrackErrorCallback(audioTrackErrorCallback)
      .createAudioDeviceModule()
  }

  private fun createAudioSource(context: Context) {
    if (mPeerConnectionFactory == null) {
      createPeerConnectionFactory(context)
    }
    mAudioSource = mPeerConnectionFactory!!.createAudioSource(MediaConstraints())
  }

  private fun createCamCapture(context: Context) {
    val isCamera2Supported = Camera2Enumerator.isSupported(context)
    val cameraEnumerator: CameraEnumerator = if (isCamera2Supported) {
      Camera2Enumerator(context)
    } else {
      Camera1Enumerator()
    }
    val deviceNames = cameraEnumerator.deviceNames
    for (deviceName in deviceNames) {
      // TODO : this needs to come from core sdk.
      val needFrontFacing = "front".endsWith("front")
      var selectedDeviceName: String? = null
      if (needFrontFacing) {
        if (cameraEnumerator.isFrontFacing(deviceName)) {
          selectedDeviceName = deviceName
        }
      } else {
        if (!cameraEnumerator.isFrontFacing(deviceName)) {
          selectedDeviceName = deviceName
        }
      }
      if (!TextUtils.isEmpty(selectedDeviceName)) {
        mCamCapture = cameraEnumerator.createCapturer(
          selectedDeviceName,
          object : CameraEventsHandler {
            override fun onCameraError(s: String) {
              utilsProvider.getControllerContainer().loggerController.traceError("camera error $s")
              utilsProvider.getControllerContainer().eventController.triggerEvent(
                DyteEventType.OnCameraError(
                  s
                )
              )
            }

            override fun onCameraDisconnected() {
              utilsProvider.getControllerContainer().loggerController.traceLog("on camera disconnected")
              utilsProvider.getControllerContainer().eventController.triggerEvent(DyteEventType.OnCameraDisconnected)
            }

            override fun onCameraFreezed(s: String) {
              utilsProvider.getControllerContainer().loggerController.traceLog("on camera freezed")
              utilsProvider.getControllerContainer().eventController.triggerEvent(
                DyteEventType.OnCameraFreezed(
                  s
                )
              )
            }

            override fun onCameraOpening(s: String) {
              utilsProvider.getControllerContainer().eventController.triggerEvent(
                DyteEventType.OnCameraOpening(
                  s
                )
              )
            }

            override fun onFirstFrameAvailable() {
              utilsProvider.getControllerContainer().eventController.triggerEvent(DyteEventType.OnFirstFrameAvailable)
            }

            override fun onCameraClosed() {
              utilsProvider.getControllerContainer().eventController.triggerEvent(DyteEventType.OnCameraClosed)
            }
          })
        break
      }
    }
    checkNotNull(mCamCapture) { "Failed to create Camera Capture" }
  }

  private fun createVideoSource(context: Context) {
    if (mPeerConnectionFactory == null) {
      createPeerConnectionFactory(context)
    }
    if (mCamCapture == null) {
      createCamCapture(context)
    }
    mVideoSource = mPeerConnectionFactory!!.createVideoSource(false)
    val surfaceTextureHelper =
      SurfaceTextureHelper.create("CaptureThread", getEglContext())
    mCamCapture!!.initialize(surfaceTextureHelper, context, mVideoSource?.capturerObserver)
    mCamCapture!!.startCapture(640, 480, 30)
  }
}