package io.dyte.core.observability

import io.dyte.core.controllers.BaseController
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.observability.LogSeverity.error
import io.dyte.core.observability.LogSeverity.log
import io.dyte.core.observability.LogSeverity.warn
import io.dyte.core.observability.LogStatus.NONE
import io.dyte.core.observability.LogStatus.PROCESSIGN
import io.dyte.core.observability.LogStatus.PUSHED
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

internal class LoggerController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), ILoggerController {
  private var shouldStart = false
  private var logs: ArrayList<DyteLog> = arrayListOf()

  private val job = SupervisorJob()
  private val scope = CoroutineScope(Dispatchers.Default + job)

  private lateinit var meetingMetaData: MeetingMetaData

  private var logId = 0

  private fun startCoroutineTimer(
    delayMillis: Long = 0,
    repeatMillis: Long = 0,
    action: () -> Unit
  ) = scope.launch(Dispatchers.Default) {
    delay(delayMillis)
    if (repeatMillis > 0) {
      while (true) {
        action()
        delay(repeatMillis)
      }
    } else {
      action()
    }
  }

  private val timer: Job = startCoroutineTimer(delayMillis = 0, repeatMillis = 7000) {
    if (!shouldStart) {
      return@startCoroutineTimer
    }
    processLogs()
  }

  private fun processLogs(shouldPushAll: Boolean = false) {
    val logs = ArrayList(logs)
    val logsToBeProcessed = logs.filter { it.status == NONE }.take(if(shouldPushAll) Int.MAX_VALUE else 20)
    if (logsToBeProcessed.isEmpty()) {
      return
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        logsToBeProcessed.map { it.status = PROCESSIGN }
        try {
          controllerContainer.apiClient.uploadLogs(
            PostDyteLogs(
              meetingMetaData,
              logsToBeProcessed,
              "mobile-core"
            )
          )
          logsToBeProcessed.forEach { processedLog ->
            logs.find { it.id == processedLog.id }!!.status = PUSHED
          }
        } catch (e:Exception) {
          logsToBeProcessed.forEach { processedLog ->
            logs.find { it.id == processedLog.id }!!.status = NONE
          }
        }
      }
    }
  }

  private fun startTimer() {
    shouldStart = true
    timer.start()
  }

  private fun cancelTimer() {
    timer.cancel()
  }

  override fun init() {
    meetingMetaData = MeetingMetaData(
      controllerContainer.metaController.getPeerId(),
      controllerContainer.metaController.getRoomName(),
      "no-org",
      controllerContainer.platformUtilsProvider.getPlatformUtils().getOsVersion(),
      true,
      controllerContainer.platformUtilsProvider.getPlatformUtils().getDeviceInfo()
    )
    startTimer()
  }

  override fun traceLog(message: String) {
    trace(message, log)
  }

  override fun traceWarning(message: String) {
    trace(message, warn)
  }

  override fun traceError(message: String) {
    trace(message, error)
  }

  private fun trace(message: String, severity: LogSeverity) {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      logs.add(
        DyteLog(
          logId,
          message,
          controllerContainer.platformUtilsProvider.getPlatformUtils().getUtcTimeNow(),
          NONE,
          severity
        )
      )
      logId++
    }
  }

  override fun pushAllNow() {
    processLogs(true)
  }

  override fun setOrgId(organizationId: String) {
    meetingMetaData = meetingMetaData.copy(organizationId = organizationId)
  }
}

@Serializable
data class DyteLog(
  val id: Int,
  val message: String,
  val loggedAt: String,
  var status: LogStatus,
  val logSeverity: LogSeverity
)


@Serializable
data class PostDyteLogs(
  val meetingMetadata: MeetingMetaData,
  val logs: List<DyteLog>,
  val serviceName: String
)

@Serializable
data class MeetingMetaData(
  @SerialName("meetingMetadata.peerId") val peerId: String,
  @SerialName("meetingMetadata.roomName") val roomName: String,
  @SerialName("meetingMetadata.organizationId") val organizationId: String,
  @SerialName("meetingMetadata.sdkVersion") val sdkVersion: String,
  @SerialName("meetingMetadata.deviceInfo.isMobile") val isMobile: Boolean,
  @SerialName("meetingMetadata.deviceInfo.deviceInfo") val deviceInfo: String
)

enum class LogStatus {
  NONE,
  PROCESSIGN,
  PUSHED;
}

enum class LogSeverity {
  info,
  error,
  debug,
  log,
  warn;
}

interface ILoggerController {
  fun traceLog(message: String)
  fun traceWarning(message: String)
  fun traceError(message: String)

  fun setOrgId(organizationId: String)

  fun pushAllNow()
}