package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.models.DyteActionResult
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.models.ParticipantFlags
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_SELF_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.UN_MUTE_SELF_AUDIO
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class SelfController(
  private val participantInfo: ParticipantInfo,
  controllerContainer: IControllerContainer
) : ISelfController, BaseController(controllerContainer) {
  private lateinit var selfParticipant: DyteSelfParticipant

  private var _roomJoined = false
  override val roomJoined: Boolean
    get() = _roomJoined

  override fun init() {
    this.selfParticipant =
      DyteSelfParticipant(
        controllerContainer.metaController.getPeerId(),
        requireNotNull(participantInfo.id),
        controllerContainer.metaController.getDisplayName(),
        null,
        false,
        participantInfo.clientSpecificId,
        ParticipantFlags(controllerContainer.presetController.canRecord(), false),
        controllerContainer
      )
  }

  override fun getSelf(): DyteSelfParticipant {
    return selfParticipant
  }

  override fun disableAudio() : DyteActionResult {
    if(_roomJoined) {
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().muteSelfAudio()
      controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
        controllerContainer.socketController.sendMessageSync(MUTE_SELF_AUDIO, null)
      }
    } else {
      selfParticipant._audioEnabled = false
    }
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
    return DyteActionResult.Success
  }

  override fun onAudioDisabled() {
    selfParticipant._audioEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override fun enableAudio() : DyteActionResult{
    if (!controllerContainer.permissionController.isPermissionGrated(MICROPHONE)) {
      return DyteActionResult.ActionNotPermitted
    }
    if (!controllerContainer.presetController.canPublishAudio()) {
      return DyteActionResult.ActionNotPermitted
    }
    if (!controllerContainer.metaController.isAudioEnabled()) {
      return DyteActionResult.ActionNotPermitted
    }

    if (_roomJoined) {
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().unmuteSelfAudio()
      controllerContainer.socketController.sendMessageSync(UN_MUTE_SELF_AUDIO, null)
    } else {
      selfParticipant._audioEnabled = true
    }
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)

    return DyteActionResult.Success
  }

  override fun onAudioEnabled() {
    selfParticipant._audioEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override fun disableVideo() : DyteActionResult {
    if (_roomJoined) {
      val producerId =
        controllerContainer.mediaSoupController.videoProduceId
      producerId?.let {
        val content = HashMap<String, JsonElement>()
        content["producerId"] = JsonPrimitive(producerId)
        controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
          controllerContainer.socketController.sendMessageSync(MUTE_VIDEO, JsonObject(content))
        }
        controllerContainer.platformUtilsProvider.getMediaSoupUtils().muteSelfVideo()
        controllerContainer.platformUtilsProvider.getPeerConnectionUtils().stopVideo()
      }
    } else {
      controllerContainer.platformUtilsProvider.getPeerConnectionUtils().stopVideo()
      selfParticipant._videoEnabled = false
    }
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
    return DyteActionResult.Success
  }

  override fun onVideoDisabled() {
    selfParticipant._videoEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun onVideoEnabled(videoProducerId: String) {
    selfParticipant._videoEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun enableVideo() : DyteActionResult {
    if (!controllerContainer.permissionController.isPermissionGrated(CAMERA)) {
      return DyteActionResult.ActionNotPermitted
    }
    if (!controllerContainer.presetController.canPublishVideo()) {
      return DyteActionResult.ActionNotPermitted
    }
    if (!controllerContainer.metaController.isVideoEnabled()) {
      return DyteActionResult.ActionNotPermitted
    }

    if (roomJoined) {
      controllerContainer.mediaSoupController.produceVideo()
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().unmuteSelfVideo()
    }
    selfParticipant._videoEnabled = true
    controllerContainer.platformUtilsProvider.getPeerConnectionUtils().resumeVideo()
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)

    return DyteActionResult.Success
  }

  override fun setDevice(dyteAndroidDevice: DyteAudioDevice) {
    controllerContainer.platformUtilsProvider.getMediaUtils().setAudioDevice(dyteAndroidDevice)
  }

  override fun getAudioDevices(): List<DyteAudioDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getAudioDevices()
  }

  override fun getSelectedAudioDevice(): DyteAudioDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedAudioDevice()
  }

  override fun onAudioDevicesChanged() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnAudioDevicesChanged)
  }

  override fun getVideoDevices(): List<DyteVideoDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getVideoDevices()
  }

  override fun setDevice(device: DyteVideoDevice) {
    controllerContainer.platformUtilsProvider.getMediaUtils().setVideoDevice(device)
  }

  override fun getSelectedVideoDevice(): DyteVideoDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedVideoDevice()
  }

  override fun onVideoDevicesChanged() {
    TODO("Not yet implemented")
  }

  override fun onEnteredInRoom() {
    _roomJoined = true
  }
}

interface ISelfController {
  val roomJoined: Boolean
  fun getSelf(): DyteSelfParticipant

  fun disableAudio(): DyteActionResult
  fun onAudioDisabled()
  fun enableAudio(): DyteActionResult
  fun onAudioEnabled()

  fun disableVideo(): DyteActionResult
  fun onVideoDisabled()
  fun enableVideo(): DyteActionResult
  fun onVideoEnabled(videoProducerId: String)

  fun getAudioDevices(): List<DyteAudioDevice>
  fun setDevice(dyteAndroidDevice: DyteAudioDevice)
  fun getSelectedAudioDevice(): DyteAudioDevice
  fun onAudioDevicesChanged()

  fun getVideoDevices(): List<DyteVideoDevice>
  fun setDevice(dye: DyteVideoDevice)
  fun getSelectedVideoDevice(): DyteVideoDevice
  fun onVideoDevicesChanged()

  fun onEnteredInRoom()
}