package io.dyte.core.controllers

import io.dyte.core.controllers.DyteRecordingState.IDLE
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.controllers.DyteRecordingState.STARTING
import io.dyte.core.controllers.DyteRecordingState.STOPPING
import io.dyte.core.models.DyteActionResult
import io.dyte.core.network.info.RecordingInfo
import io.dyte.core.network.models.RecordingData
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlin.Exception

internal class RecordingController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), IRecordingController {
  private lateinit var recordingInfo: RecordingInfo
  private var recordingState: DyteRecordingState = IDLE

  override fun init() {
    println("DyteMobileClient | RecordingController init ")
    try {
      recordingInfo = runBlocking(Dispatchers.Default) {
        controllerContainer.apiClient.getActiveRecording()
      }
      updateState(RECORDING)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
    } catch (e:Exception) {
      println("DyteMobileClient | RecordingController init ${e.message}")
      e.printStackTrace()
    }
  }

  override suspend fun start() : DyteActionResult{
    if (controllerContainer.presetController.canRecord().not()) {
      return DyteActionResult.ActionNotPermitted
    }

    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        try {
          updateState(STARTING)
          recordingInfo = controllerContainer.apiClient.startRecording()
        } catch (e:Exception) {
          e.printStackTrace()
          controllerContainer.loggerController.traceError("error in starting recording ${e.message}")
          updateState(IDLE)
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
        }
      }
    }
    return DyteActionResult.Success
  }

  override suspend fun stop() : DyteActionResult{
    if (controllerContainer.presetController.canRecord().not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        try {
          updateState(STOPPING)
          controllerContainer.apiClient.stopRecording(recordingInfo)
        } catch (e:Exception) {
          e.printStackTrace()
          controllerContainer.loggerController.traceError("error in stopping recording ${e.message}")
          updateState(IDLE)
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
        }
      }
    }
    return DyteActionResult.Success
  }

  override fun getState(): DyteRecordingState {
    return recordingState
  }

  override fun fetchRecordingData() {
    recordingInfo = runBlocking(Dispatchers.Default) {
      controllerContainer.apiClient.getActiveRecording()
    }
  }

  private fun updateState(newState: DyteRecordingState) {
    recordingState = newState
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStateUpdate(recordingState))
  }

  override fun onRecordingPeerJoined() {
    updateState(RECORDING)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
  }

  override fun onRecordingPeerLeft() {
    updateState(IDLE)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStopped)
  }
}

interface IRecordingController {
  suspend fun start(): DyteActionResult
  suspend fun stop(): DyteActionResult
  fun getState(): DyteRecordingState

  fun onRecordingPeerJoined()
  fun onRecordingPeerLeft()

  fun fetchRecordingData()
}

enum class DyteRecordingState {
  IDLE,
  STARTING,
  RECORDING,
  STOPPING;
}