package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.controllers.PermissionType.STORAGE

internal class PermissionController(controller: IControllerContainer) : IPermissionController,
  BaseController(controller) {
  override fun init() {
  }

  override val permissions: List<Permission>
    get() = _permissions.toList()

  private val _permissions = arrayListOf<Permission>()

  private var isNotified = false

  override fun isPermissionGrated(permissionType: PermissionType): Boolean {
    return _permissions.find { it.type == permissionType }?.isGranted ?: false
  }

  override fun askPermissions(
    list: List<PermissionType>,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  ) {
    _permissions.clear()
    list.forEach {
      _permissions.add(Permission(it, false))
    }

    controllerContainer.platformUtilsProvider.getPlatformUtils()
      .askPermissions(
        { permType ->
          val permission = requireNotNull(_permissions.find { it.type == permType })
          permission.isGranted = true
          if ( isNotified.not()) {
            onPermissionGranted.invoke()
            isNotified = true
          }
        }, { permType ->
          val permission = requireNotNull(_permissions.find { it.type == permType })
          permission.isGranted = false
          if ( isNotified.not()) {
            onPermissionDenied.invoke()
            isNotified = true
          }
        })
  }

  override fun arePermissionsGranted() = _permissions.find { it.isGranted.not() } == null

  override fun processPermissionForSelf() {
    _permissions.forEach { permission ->
      when (permission.type) {
        CAMERA -> {
          controllerContainer.selfController.getSelf()._cameraPermissionGranted = permission.isGranted
        }
        MICROPHONE -> {
          controllerContainer.selfController.getSelf()._microphonePermissionGranted = permission.isGranted
        }
        STORAGE -> {

        }
      }
    }
  }
}

interface IPermissionController {
  val permissions: List<Permission>

  fun arePermissionsGranted(): Boolean

  fun isPermissionGrated(permissionType: PermissionType): Boolean

  fun processPermissionForSelf()

  fun askPermissions(
    list: List<PermissionType>,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  )
}

data class Permission(
  val type: PermissionType,
  var isGranted: Boolean
)

enum class PermissionType(val displayName: String) {
  CAMERA("Camera"),
  MICROPHONE("Mic"),
  STORAGE("Storage");
}