package io.dyte.core.controllers

import io.dyte.core.models.DyteMeta
import io.dyte.core.network.info.ParticipantInfo

/**
 * Meta controller
 *
 * Contains meta details about the meeting,
 * example: [DyteMeetingInfoV1]
 *
 * @constructor Create empty Meta controller
 */
internal class MetaController(
  private val meetingInfo: MeetingInfo,
  controllerContainer: IControllerContainer
) : IMetaController, BaseController(controllerContainer) {
  private lateinit var participantInfo: ParticipantInfo
  private lateinit var meetingUserPeerId: String
  private lateinit var dyteMeta: DyteMeta

  override fun init() {
    meetingUserPeerId = controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid()
    dyteMeta = DyteMeta(controllerContainer)
    meetingInfo.meetingId = controllerContainer.platformUtilsProvider.getPlatformUtils().decodeAuthToken(meetingInfo.authToken)
    meetingInfo.isV2Meeting = meetingInfo.meetingId != null
  }

  override fun getDisplayName(): String {
    return participantInfo.name
  }

  override fun getRoomName(): String {
    return meetingInfo.getRoomName()
  }

  override fun getAuthToken(): String {
    return meetingInfo.authToken
  }

  override fun getBaseUrl(): String {
    return meetingInfo.getBaseUrl()
  }

  override fun getPeerId(): String {
    return meetingUserPeerId
  }
  
  override fun getOrgId(): String {
    return participantInfo.organizationId
  }

  override fun getMeetingTitle(): String {
    return meetingInfo.meetingTitle ?: ""
  }

  override fun getMeetingStatedTimestamp(): String {
    return meetingInfo.meetingStartedAt ?: ""
  }

  override fun getRoomType(): String {
    return participantInfo.presetInfo.viewType
  }

  override fun setMeetingTitle(title: String) {
    meetingInfo.meetingTitle = title
  }

  override fun setMeetingStartedTimestamp(timestamp: String) {
    meetingInfo.meetingStartedAt = timestamp
  }

  override fun getMeetingId(): String {
    return meetingInfo.getRoomName()
  }

  override fun isV2AuthToken(): Boolean {
    return meetingInfo.isV2Meeting
  }

  override fun setParticipantInfo(participantInfo: ParticipantInfo) {
    this.participantInfo = participantInfo
  }

  override fun isWebinar(): Boolean {
    return participantInfo.presetInfo.viewType == "WEBINAR"
  }

  override fun isGroupCall(): Boolean {
    return participantInfo.presetInfo.viewType == "GROUP_CALL"
  }

  override fun isAudioEnabled(): Boolean {
    return meetingInfo.config.enableAudio
  }

  override fun isVideoEnabled(): Boolean {
    return meetingInfo.config.enableAudio
  }
}

interface IMetaController {
  fun getPeerId(): String
  fun getBaseUrl(): String
  fun getDisplayName(): String
  fun getRoomName(): String
  fun getAuthToken(): String
  fun getOrgId(): String

  fun setMeetingTitle(title: String)
  fun setMeetingStartedTimestamp(timestamp: String)

  fun getMeetingTitle(): String
  fun getMeetingStatedTimestamp(): String
  fun getRoomType(): String

  fun getMeetingId(): String

  fun isWebinar(): Boolean
  fun isGroupCall(): Boolean

  fun isAudioEnabled(): Boolean

  fun isVideoEnabled(): Boolean

  fun setParticipantInfo(participantInfo: ParticipantInfo)

  fun isV2AuthToken(): Boolean
}

