package io.dyte.core.controllers

import io.dyte.core.models.DyteActionResult
import io.dyte.core.models.DyteMeetingParticipant
import io.dyte.core.socket.events.OutboundMeetingEventType.ACCEPT_WAITLIST_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.KICK_ALL
import io.dyte.core.socket.events.OutboundMeetingEventType.KICK_PEER
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_ALL_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_ALL_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_PEER_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_PEER_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.PIN_PARTICIPANT
import io.dyte.core.socket.events.OutboundMeetingEventType.REJECT_WAITLIST_REQUEST
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class HostController(controllerContainer: IControllerContainer) : IHostController,
  BaseController(controllerContainer) {
  override fun init() {
  }

  override fun kick(id: String) : DyteActionResult{
    if (controllerContainer.presetController.permissions.hostPermissions.canKickParticipant.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(id)
      controllerContainer.socketController.sendMessageSync(KICK_PEER, JsonObject(content))
    }
    return DyteActionResult.Success
  }

  // TODO : is there any differance between kick single and kick all ?
  override fun kickAll() : DyteActionResult{
    if (controllerContainer.presetController.permissions.hostPermissions.canKickParticipant.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      controllerContainer.socketController.sendMessageSync(KICK_ALL, null)
    }
    return DyteActionResult.Success
  }

  // TODO : is there any differance between mute single and mute all ?
  override fun muteAllAudio(allowUnmute: Boolean) : DyteActionResult{
    if (controllerContainer.presetController.permissions.hostPermissions.canDisableParticipantAudio.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      content["allowUnMute"] = JsonPrimitive(allowUnmute)
      controllerContainer.socketController.sendMessageSync(MUTE_ALL_AUDIO, JsonObject(content))
    }
    return DyteActionResult.Success
  }

  override fun muteAllVideo() : DyteActionResult{
    if (controllerContainer.presetController.permissions.hostPermissions.canDisableParticipantVideo.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      controllerContainer.socketController.sendMessageSync(MUTE_ALL_VIDEO, null)
    }
    return DyteActionResult.Success
  }

  override fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant) : DyteActionResult{
    if (controllerContainer.presetController.permissions.hostPermissions.canDisableParticipantAudio.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessageSync(MUTE_PEER_AUDIO, JsonObject(content))
    }
    return DyteActionResult.Success
  }

  override fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant) : DyteActionResult{
    if (controllerContainer.presetController.permissions.hostPermissions.canDisableParticipantVideo.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessageSync(MUTE_PEER_VIDEO, JsonObject(content))
    }
    return DyteActionResult.Success
  }

  override fun acceptAllWaitingRequests() :DyteActionResult {
    if (controllerContainer.presetController.permissions.hostPermissions.acceptWaitingRequests.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      controllerContainer.participantController.meetingRoomParticipants.waitlisted.forEach {
        acceptWaitlistedRequest(it)
      }
    }
    return DyteActionResult.Success
  }

  override fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant) :DyteActionResult {
    if (controllerContainer.presetController.permissions.hostPermissions.acceptWaitingRequests.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessageSync(ACCEPT_WAITLIST_REQUEST, JsonObject(content))
    }
    return DyteActionResult.Success
  }

  override fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant) :DyteActionResult {
    if (controllerContainer.presetController.permissions.hostPermissions.acceptWaitingRequests.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessageSync(REJECT_WAITLIST_REQUEST, JsonObject(content))
    }
    return DyteActionResult.Success
  }

  override fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant) :DyteActionResult{
    if (controllerContainer.presetController.permissions.hostPermissions.canPinParticipant.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessageSync(PIN_PARTICIPANT, JsonObject(content))
    }
    return DyteActionResult.Success
  }

  override fun unpinParticipant() :DyteActionResult {
    if (controllerContainer.presetController.permissions.hostPermissions.canPinParticipant.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      val content = HashMap<String, JsonElement>()
      val nullString : String? = null
      content["id"] = JsonPrimitive(nullString)
      controllerContainer.socketController.sendMessageSync(PIN_PARTICIPANT, JsonObject(content))
    }
    return DyteActionResult.Success
  }
}

interface IHostController {
  fun muteAllAudio(allowUnmute: Boolean): DyteActionResult
  fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant): DyteActionResult

  fun muteAllVideo(): DyteActionResult
  fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant): DyteActionResult

  fun kickAll(): DyteActionResult
  fun kick(id: String): DyteActionResult

  fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant): DyteActionResult
  fun unpinParticipant(): DyteActionResult

  fun acceptAllWaitingRequests(): DyteActionResult
  fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant): DyteActionResult
  fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant): DyteActionResult
}
