package io.dyte.core.platform

import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.content.pm.PackageManager
import android.os.Build
import android.os.Build.VERSION
import android.os.Build.VERSION_CODES
import android.os.CountDownTimer
import android.util.Base64
import android.util.Log
import androidx.annotation.RequiresApi
import io.dyte.core.controllers.PermissionType
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import org.json.JSONObject
import java.io.File
import java.io.UnsupportedEncodingException
import java.net.URLEncoder
import java.text.DateFormat
import java.text.SimpleDateFormat
import java.util.Date
import java.util.TimeZone
import java.util.UUID

/**
 * Dyte android platform
 *
 * Place to put misc platform specific code
 *
 * @constructor Create empty Dyte android platform
 */
internal class DyteAndroidPlatform(
  private val activity: Activity,
  private val utilsProvider: IDytePlatformUtilsProvider
) :
  IDytePlatformUtils {

  /**
   * Get current time
   *
   * @return current time in millis
   */
  override fun getCurrentTime(): Long {
    return System.currentTimeMillis()
  }

  /**
   * Get uuid
   *
   * @return random uuid
   */
  override fun getUuid(): String {
    return UUID.randomUUID().toString()
  }

  /**
   * Run on main thread
   *
   * runs code on main thread
   *
   * @param block
   * @receiver
   */
  @OptIn(DelicateCoroutinesApi::class)
  override fun runOnMainThread(block: () -> Unit) {
    GlobalScope.launch(Dispatchers.Main) {
      block.invoke()
    }
  }

  /**
   * Run on io thread
   *
   * runs code on io thread
   *
   * @param block
   * @receiver
   */
  @OptIn(DelicateCoroutinesApi::class)
  override fun runOnIoThread(block: () -> Unit) {
    GlobalScope.launch(Dispatchers.IO) {
      block.invoke()
    }
  }

  override fun getAndroidApplicationContext(): Any {
    return activity.application
  }

  override fun getActivity(): Any {
    return activity
  }

  // "yyyy-MM-dd'T'HH:mm:ss'Z'"
  @SuppressLint("SimpleDateFormat")
  override fun getDiff(startTime: String, endTime: String): String {
    val dates = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
    val date1 = dates.parse(startTime)
    val date2 = dates.parse(endTime)
    val differenceInMillis = Math.abs(date1.time - date2.time).toInt()

    val seconds: Int = differenceInMillis / 1000 % 60
    val minutes: Int = (differenceInMillis - seconds) / 1000 / 60

    var newSeconds = seconds.toString()
    if (seconds < 10) {
      newSeconds = "0$newSeconds"
    }

    var newMinutes = minutes.toString()
    if (minutes < 10) {
      newMinutes = "0$newMinutes"
    }

    return "$newMinutes:$newSeconds"
  }

  // "YYYY-MM-DDThh:mm:ss.sTZD"
  override fun getUtcTimeNow(): String {
    val m_ISO8601Local: DateFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
    m_ISO8601Local.timeZone = TimeZone.getTimeZone("UTC")
    return m_ISO8601Local.format(Date())
  }

  override fun getUserDisplayableTime(time: Long): String {
    val dateFormat: DateFormat = SimpleDateFormat("HH:mm")
    return dateFormat.format(Date(time))
  }

  override fun getFileContent(filePath: String): ByteArray {
    return File(filePath).readBytes()
  }

  override fun printThread(desc: String?) {
    Log.v("DyteMobileClient", "current thread ${Thread.currentThread().name} for $desc")
  }

  @RequiresApi(VERSION_CODES.M)
  override fun askPermissions(
    onPermissionGranted: (permission: PermissionType) -> Unit,
    onPermissionsDenied: (permission: PermissionType) -> Unit
  ) {
    processPermissions(true, onPermissionGranted, onPermissionsDenied)
  }

  @SuppressLint("NewApi")
  private fun pollForPermissions(
    onPermissionGranted: (permission: PermissionType) -> Unit,
    onPermissionsDenied: (permission: PermissionType) -> Unit
  ) {
    // timer for 10 seconds, ticked every 1 second
    val timer = object : CountDownTimer(10000, 1000) {
      override fun onTick(millisUntilFinished: Long) {
        processPermissions(false, onPermissionGranted, onPermissionsDenied)
      }

      override fun onFinish() {
        processPermissions(false, onPermissionGranted, onPermissionsDenied)

        val permissionsToTask = arrayListOf<String>()
        if (utilsProvider.getControllerContainer().metaController.isAudioEnabled()) {
          permissionsToTask.add(Manifest.permission.RECORD_AUDIO)
        }
        if (utilsProvider.getControllerContainer().metaController.isVideoEnabled()) {
          permissionsToTask.add(Manifest.permission.CAMERA)
        }

        if (VERSION.SDK_INT < VERSION_CODES.M) {
          utilsProvider.getControllerContainer().permissionController.permissions.forEach {
            onPermissionGranted.invoke(it.type)
          }
          return
        }

        val notGrantedPermTypes = arrayListOf<PermissionType>()
        val grantedPerms = arrayListOf<PermissionType>()

        permissionsToTask.forEach { permission ->
          if (activity.checkSelfPermission(permission) != PackageManager.PERMISSION_GRANTED) {
            if (permission == Manifest.permission.RECORD_AUDIO) {
              notGrantedPermTypes.add(MICROPHONE)
            }
            if (permission == Manifest.permission.CAMERA) {
              notGrantedPermTypes.add(CAMERA)
            }
          }

          if (activity.checkSelfPermission(permission) == PackageManager.PERMISSION_GRANTED) {
            if (permission == Manifest.permission.RECORD_AUDIO) {
              grantedPerms.add(MICROPHONE)
            }
            if (permission == Manifest.permission.CAMERA) {
              grantedPerms.add(CAMERA)
            }
          }
        }

        grantedPerms.forEach {
          onPermissionGranted.invoke(it)
        }

        notGrantedPermTypes.forEach {
          onPermissionsDenied.invoke(it)
        }
      }
    }
    timer.start()
  }

  @RequiresApi(VERSION_CODES.M)
  private fun processPermissions(
    shouldAsk: Boolean,
    onPermissionGranted: (permission: PermissionType) -> Unit,
    onPermissionsDenied: (permission: PermissionType) -> Unit
  ) {
    val permissionsToTask = arrayListOf<String>()
    val permTypesToAsk = arrayListOf<PermissionType>()

    if (utilsProvider.getControllerContainer().metaController.isAudioEnabled()) {
      permissionsToTask.add(Manifest.permission.RECORD_AUDIO)
      permTypesToAsk.add(MICROPHONE)
    }
    if (utilsProvider.getControllerContainer().metaController.isVideoEnabled()) {
      permissionsToTask.add(Manifest.permission.CAMERA)
      permTypesToAsk.add(CAMERA)
    }

    if (permissionsToTask.isEmpty()) {
      onPermissionGranted.invoke(CAMERA)
    }

    if (VERSION.SDK_INT < VERSION_CODES.M) {
      utilsProvider.getControllerContainer().permissionController.permissions.forEach {
        onPermissionGranted.invoke(it.type)
      }
      return
    }

    val notGrantedPermTypes = arrayListOf<PermissionType>()
    permissionsToTask.forEach { permission ->
      if (activity.checkSelfPermission(permission) != PackageManager.PERMISSION_GRANTED) {
        if (permission == Manifest.permission.RECORD_AUDIO) {
          notGrantedPermTypes.add(MICROPHONE)
        }
        if (permission == Manifest.permission.CAMERA) {
          notGrantedPermTypes.add(CAMERA)
        }
      }
    }

    if (notGrantedPermTypes.isEmpty()) {
      permTypesToAsk.forEach {
        onPermissionGranted.invoke(it)
      }
      return
    }

    if (shouldAsk) {
      activity.requestPermissions(permissionsToTask.toTypedArray(), 111)
      pollForPermissions(onPermissionGranted, onPermissionsDenied)
    }
  }


  override fun listenForCrashes() {
    val defaultHandler = Thread.getDefaultUncaughtExceptionHandler()
    Thread.setDefaultUncaughtExceptionHandler { thread, throwable ->
      utilsProvider.getControllerContainer().loggerController.traceError(throwable.message ?: "crash happened, reason unknown")
      utilsProvider.getControllerContainer().loggerController.pushAllNow()
      if (defaultHandler != null) {
        defaultHandler.uncaughtException(thread, throwable)
      }
    }
  }

  override fun getOsName(): String {
    return "android"
  }

  override fun getOsVersion(): String {
    return VERSION.SDK_INT.toString()
  }

  override fun getDeviceInfo(): String {
    return Build.MODEL + " | " + Build.MANUFACTURER
  }

  override fun decodeAuthToken(authToken: String): String? {
    var meetingId: String? = null
    try {
      val split = authToken.split("\\.".toRegex()).toTypedArray()
      val body = JSONObject(getJson(split[1]))
      if (body.has("meetingId")) {
        meetingId = body.getString("meetingId")
      }
    } catch (e: UnsupportedEncodingException) {
      throw IllegalArgumentException("Invalid auth token")
    }
    return meetingId
  }

  @Throws(UnsupportedEncodingException::class)
  private fun getJson(strEncoded: String): String {
    val decodedBytes: ByteArray = Base64.decode(strEncoded, Base64.URL_SAFE)
    return io.ktor.utils.io.core.String(decodedBytes)
  }

  override fun getUrlEncodedString(stringToEncode: String): String {
    return URLEncoder.encode(stringToEncode, "UTF-8")
  }
}