package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnWebinarStoppedPresenting
import io.dyte.core.socket.events.OutboundMeetingEventType.ACCEPT_PRESENTING_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.REJECT_PRESENTING_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.REQUEST_TO_JOIN_STAGE
import io.dyte.core.socket.events.OutboundMeetingEventType.START_PRESENTING
import io.dyte.core.socket.events.OutboundMeetingEventType.STOP_PRESENTING
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject

internal class WebinarController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), IWebinarController {
  private var isPresenting = false
  private var requestsToPresent = arrayListOf<RequestToPresentParticipant>()

  override fun init() {
    // no-op
  }

  override fun joinStage() {
    controllerContainer.socketController.sendMessageSync(
      REQUEST_TO_JOIN_STAGE,
      JsonPrimitive("REQUEST_TO_PRESENT")
    )
  }

  override fun leaveStage() {
    controllerContainer.socketController.sendMessageSync(STOP_PRESENTING, null)
  }

  override fun onRequestedToPresent(requestType: String) {
    if (requestType == "REQUEST_TO_PRESENT") {
      controllerContainer.eventController.triggerEvent(DyteEventType.OnWebinarPresentRequest)
    }
  }

  override fun acceptRequestToPresent() {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      isPresenting = true
      controllerContainer.socketController.sendMessageSync(START_PRESENTING, null)
      val selfTracks = controllerContainer.platformUtilsProvider.getMediaSoupUtils().getSelfTrack()
      val selfParticipant = controllerContainer.selfController.getSelf()
      // selfParticipant.audioTrack = selfTracks.first
      selfParticipant._videoTrack = selfTracks.second
      controllerContainer.participantController.onParticipantStreamConnected(selfParticipant)
      selfParticipant.disableVideo()
      selfParticipant.disableAudio()
    }
  }

  override fun rejectRequestToPresent() {
    isPresenting = false
    controllerContainer.socketController.sendMessageSync(STOP_PRESENTING, null)
  }

  override fun onPeerAddedToStage(peerId: String) {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      requestsToPresent.removeAll { it.id == peerId }
      if (peerId == controllerContainer.selfController.getSelf().id) {
        controllerContainer.roomNodeController.connectProducerTransport()
      }
    }
  }

  override fun onPeerRejectedToStage(peerId: String) {
    requestsToPresent.removeAll { it.id == peerId }
  }

  override fun isPresenting(): Boolean {
    return isPresenting
  }

  override fun onStoppedPresenting() {
    isPresenting = false
    controllerContainer.eventController.triggerEvent(OnWebinarStoppedPresenting)
  }

  override fun onRequestToPresentPeerAdded(requestToPresentParticipant: RequestToPresentParticipant) {
    requestsToPresent.add(requestToPresentParticipant)
  }

  override fun acceptRequest(id: String) {
    val arrayIds = buildJsonArray {
      addJsonObject {
        put("id", JsonPrimitive(id))
      }
    }
    controllerContainer.socketController.sendMessageSync(ACCEPT_PRESENTING_REQUEST, arrayIds)
  }

  override fun acceptAllRequest() {
    val arrayIds = buildJsonArray {
      requestsToPresent.forEach {
        addJsonObject {
          put("id", JsonPrimitive(it.id))
        }
      }
    }
    controllerContainer.socketController.sendMessageSync(ACCEPT_PRESENTING_REQUEST, arrayIds)
  }

  override fun rejectRequest(id: String) {
    val idToReject = buildJsonObject {
      put("id", JsonPrimitive(id))
    }
    controllerContainer.socketController.sendMessageSync(REJECT_PRESENTING_REQUEST, idToReject)
  }

  override fun rejectAllRequest() {
    requestsToPresent.forEach {
      val idToReject = buildJsonObject {
        put("id", JsonPrimitive(it.id))
      }
      controllerContainer.socketController.sendMessageSync(REJECT_PRESENTING_REQUEST, idToReject)
    }
  }
}

interface IWebinarController {
  fun joinStage()
  fun leaveStage()

  fun onRequestedToPresent(requestType: String)
  fun acceptRequestToPresent()
  fun rejectRequestToPresent()

  fun onRequestToPresentPeerAdded(requestToPresentParticipant: RequestToPresentParticipant)

  fun acceptRequest(id: String)
  fun acceptAllRequest()
  fun rejectRequest(id: String)
  fun rejectAllRequest()

  fun isPresenting(): Boolean

  fun onPeerAddedToStage(peerId: String)
  fun onPeerRejectedToStage(peerId: String)
  fun onStoppedPresenting()
}

@kotlinx.serialization.Serializable
data class RequestToPresentParticipant(
  val id: String,
  val name: String,
  val requestToJoinType: String
)