package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.models.WaitListStatus.WAITING
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.CREATE_WEB_RTC_TRANSPORT
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROUTER_RTP_CAPABILITIES
import io.dyte.core.socket.events.payloadmodel.inbound.Device
import io.dyte.core.socket.events.payloadmodel.outbound.CreateWebRTCTransportPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.JoinRoomPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.encodeToJsonElement

internal class RoomNodeController(
  controllerContainer: IControllerContainer
) : IRoomNodeController, BaseController(controllerContainer) {

  private lateinit var iceServers: List<IceServerData>
  private lateinit var jsonParser: Json

  private lateinit var roomState: WebSocketRoomStateModel
  private lateinit var routerCapabilities: RouterCapabilitiesModel
  private lateinit var webRtcCreateTransportModelConsumer: WebRtcCreateTransportModel
  private lateinit var webRtcCreateTransportModelProducer: WebRtcCreateTransportModel
  private lateinit var joinRoom: WebSocketJoinRoomModel

  override fun init() {
    jsonParser = BaseApiService.json

    try {
      iceServers = runBlocking(Dispatchers.Default) { getICEServers() }
    } catch (e: Exception) {
      controllerContainer.loggerController.traceError("error in getting ice servers ${e.message}")
      return
    }
  }

  override fun joinRoom() {
    controllerContainer.loggerController.traceLog("join room started")
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)
    try {
      val roomStateResponse =
        controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
      roomState =
        requireNotNull(
          controllerContainer.socketMessageResponseParser.parseResponse(
            roomStateResponse
          ).payload as WebSocketRoomStateModel
        )
      controllerContainer.loggerController.traceLog("room state received")

      // check for remote peers joined before local user
      controllerContainer.participantController.handleRoomState(roomState)
      // Pick out active plugins, if any
      controllerContainer.pluginsController.handleRoomState(roomState)

      controllerContainer.loggerController.traceLog("room state processed")

      val routerCapabilitiesResponse =
        controllerContainer.socketController.sendMessageSync(GET_ROUTER_RTP_CAPABILITIES, null)
      routerCapabilities =
        (controllerContainer.socketMessageResponseParser.parseResponse(routerCapabilitiesResponse).payload as RouterCapabilitiesModel)

      controllerContainer.loggerController.traceLog("received and processed router capabilities")

      controllerContainer.platformUtilsProvider.getMediaSoupUtils()
        .loadRouterRtpCapabilities(jsonParser.encodeToString(routerCapabilities))
      controllerContainer.loggerController.traceLog("loaded router rtp capabilities into media soup device")

      val consumerResponse =
        controllerContainer.socketController.sendMessageSync(
          CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
            CreateWebRTCTransportPayloadRequestModel(
              forceTcp = false,
              producing = false,
              consuming = true
            )
          )
        )

      webRtcCreateTransportModelConsumer =
        (controllerContainer.socketMessageResponseParser.parseResponse(consumerResponse).payload as WebRtcCreateTransportModel)
      controllerContainer.platformUtilsProvider.getMediaSoupUtils()
        .createWebRtcTransportRecv(webRtcCreateTransportModelConsumer, iceServers)
      controllerContainer.loggerController.traceLog("created receive transport")

      val producerResponse =
        controllerContainer.socketController.sendMessageSync(
          CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
            CreateWebRTCTransportPayloadRequestModel(
              forceTcp = false,
              producing = true,
              consuming = false
            )
          )
        )
      webRtcCreateTransportModelProducer =
        controllerContainer.socketMessageResponseParser.parseResponse(producerResponse).payload as WebRtcCreateTransportModel
      controllerContainer.platformUtilsProvider.getMediaSoupUtils()
        .createWebRtcTransportProd(webRtcCreateTransportModelProducer, iceServers)
      controllerContainer.loggerController.traceLog("created produce transport")

      val joinRoomPayload = JoinRoomPayloadRequestModel()
      joinRoomPayload.device = Device(isMobile = true, sdkName = "mobile-core", osName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName(), osVersionName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsVersion())
      joinRoomPayload.displayName = controllerContainer.selfController.getSelf().name
      joinRoomPayload.rtpCapabilities = routerCapabilities
      val payload = jsonParser.encodeToJsonElement(joinRoomPayload)
      val joinRoomResponse =
        controllerContainer.socketController.sendMessageSync(
          OutboundMeetingEventType.JOIN_ROOM,
          payload
        )
      joinRoom =
        controllerContainer.socketMessageResponseParser.parseResponse(joinRoomResponse).payload as WebSocketJoinRoomModel

      controllerContainer.loggerController.traceLog("join room done!")

      if (joinRoom.waitlisted == true) {
        controllerContainer.loggerController.traceLog("aww snap, I'm in waiting room!")
        controllerContainer.selfController.getSelf()._waitListStatus = WAITING
        controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfWaitListStatusUpdate(WAITING))
        return
      }

      controllerContainer.participantController.handleRoomJoined(joinRoom)
      controllerContainer.metaController.setMeetingStartedTimestamp(requireNotNull(joinRoom.startedAt))

      controllerContainer.loggerController.traceLog("can produce audio ? ${controllerContainer.presetController.canPublishAudio()} } video ? ${controllerContainer.presetController.canPublishVideo()}")

      if (controllerContainer.presetController.permissions.hostPermissions.canPresent && (controllerContainer.presetController.canPublishAudio() || controllerContainer.presetController.canPublishVideo())) {
        connectProducerTransport()
      }

      val selfTracks = controllerContainer.platformUtilsProvider.getMediaSoupUtils().getSelfTrack()
      val selfParticipant = controllerContainer.selfController.getSelf()
      // selfParticipant.audioTrack = selfTracks.first
      selfParticipant._videoTrack = selfTracks.second
      controllerContainer.participantController.onSelfJoined()

      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnMeetingRoomJoined(
          controllerContainer.selfController.getSelf()
        )
      )
      controllerContainer.loggerController.traceLog("joined!")

      if (controllerContainer.permissionController.isPermissionGrated(CAMERA).not()) {
        controllerContainer.loggerController.traceLog("joined without camera permission")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
        )
      }
      if (controllerContainer.permissionController.isPermissionGrated(MICROPHONE).not()) {
        controllerContainer.loggerController.traceLog("joined without microphone permission")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
        )
      }

      controllerContainer.pollsController.loadPolls()
      controllerContainer.chatController.loadChatMessages()
    } catch (e:Exception) {
      e.printStackTrace()
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
    }
  }

  override fun connectProducerTransport() {
    println("DyteMobileClient | RoomNodeController connectProducerTransport ")
    controllerContainer.platformUtilsProvider.getMediaSoupUtils()
      .connectTransport(requireNotNull(webRtcCreateTransportModelConsumer.id), true) {

      }
  }

  private suspend fun getICEServers(): List<IceServerData> {
    return requireNotNull(controllerContainer.apiClient.getICEServers().iceServers)
  }
}

interface IRoomNodeController {
  fun joinRoom()
  fun connectProducerTransport()
}