package io.dyte.core.controllers

import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeta
import io.dyte.core.network.info.ParticipantInfo
import io.ktor.utils.io.charsets.Charsets
import io.ktor.utils.io.core.String

/**
 * Meta controller
 *
 * Contains meta details about the meeting,
 * example: [DyteMeetingInfo]
 *
 * @constructor Create empty Meta controller
 */
internal class MetaController(
  private val dyteMeetingInfo: DyteMeetingInfo,
  controllerContainer: IControllerContainer
) : IMetaController, BaseController(controllerContainer) {
  private lateinit var participantInfo: ParticipantInfo
  private lateinit var meetingUserPeerId: String
  private lateinit var meetingStartedTimestamp: String
  private lateinit var meetingTitle: String

  private lateinit var dyteMeta: DyteMeta

  private var _meetingId: String? = null

  private var isV2AuthToken = false

  override fun init() {
    meetingUserPeerId = controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid()
    dyteMeta = DyteMeta(controllerContainer)
    _meetingId = controllerContainer.platformUtilsProvider.getPlatformUtils().decodeAuthToken(dyteMeetingInfo.authToken)
    isV2AuthToken = _meetingId != null
  }

  override fun getMeetingInfo(): DyteMeetingInfo {
    return dyteMeetingInfo
  }

  override fun getDisplayName(): String {
    return participantInfo.name
  }

  override fun getRoomName(): String {
    return dyteMeetingInfo.roomName
  }

  override fun getAuthToken(): String {
    return dyteMeetingInfo.authToken
  }

  override fun getBaseUrl(): String {
    return dyteMeetingInfo.baseUrl
  }

  override fun getPeerId(): String {
    return meetingUserPeerId
  }
  
  override fun getOrgId(): String {
    return participantInfo.organizationId
  }

  override fun getMeetingTitle(): String {
    return meetingTitle
  }

  override fun getMeetingStatedTimestamp(): String {
    return meetingStartedTimestamp
  }

  override fun getRoomType(): String {
    return participantInfo.presetInfo.viewType
  }

  override fun setMeetingTitle(title: String) {
    this.meetingTitle = title
  }

  override fun setMeetingStartedTimestamp(timestamp: String) {
    this.meetingStartedTimestamp = timestamp
  }

  override fun getMeetingId(): String {
    return if(isV2AuthToken) {
      requireNotNull(_meetingId)
    } else {
      getRoomName()
    }
  }

  override fun isV2AuthToken(): Boolean {
    return isV2AuthToken
  }

  override fun setParticipantInfo(participantInfo: ParticipantInfo) {
    this.participantInfo = participantInfo
  }

  override fun isWebinar(): Boolean {
    return participantInfo.presetInfo.viewType == "WEBINAR"
  }

  override fun isGroupCall(): Boolean {
    return participantInfo.presetInfo.viewType == "GROUP_CALL"
  }
}

interface IMetaController {
  fun getMeetingInfo(): DyteMeetingInfo

  fun getPeerId(): String
  fun getBaseUrl(): String
  fun getDisplayName(): String
  fun getRoomName(): String
  fun getAuthToken(): String
  fun getOrgId(): String

  fun setMeetingTitle(title: String)
  fun setMeetingStartedTimestamp(timestamp: String)

  fun getMeetingTitle(): String
  fun getMeetingStatedTimestamp(): String
  fun getRoomType(): String

  fun getMeetingId(): String

  fun isWebinar(): Boolean
  fun isGroupCall(): Boolean

  fun setParticipantInfo(participantInfo: ParticipantInfo)

  fun isV2AuthToken(): Boolean
}

