package io.dyte.core.network

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.network.info.MeetingSessionInfo
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.network.info.RecordingInfo
import io.dyte.core.network.models.GraphQlRequest
import io.dyte.core.network.models.GraphQlRequestVariables
import io.dyte.core.network.models.IceServersWrapper
import io.dyte.core.network.models.MeetingSessionDataResponseV1
import io.dyte.core.network.models.MeetingSessionDataWrapper
import io.dyte.core.network.models.MultiplePluginResponse
import io.dyte.core.network.models.PluginAuthResponse
import io.dyte.core.network.models.PluginAuthorizationModel
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.network.models.PluginResponse
import io.dyte.core.network.models.RecordingData
import io.dyte.core.network.models.RecordingResponseWrapper
import io.dyte.core.network.models.RecordingResponseWrapperV2
import io.dyte.core.network.models.StartRecordingModel
import io.dyte.core.network.models.StopRecordingModel
import io.dyte.core.network.models.StopRecordingModelV2
import io.dyte.core.network.models.UserDataWrapper
import io.dyte.core.network.models.UserDataWrapperV1
import io.dyte.core.network.models.UserPresetDataWrapper
import io.dyte.core.network.models.UserPresetRequestModel
import io.dyte.core.observability.PostDyteLogs
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.plugins.logging.LogLevel.ALL
import io.ktor.client.plugins.logging.LogLevel.NONE
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.plugins.logging.SIMPLE
import io.ktor.client.request.get
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.http.ContentType
import io.ktor.http.contentType
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json

// TODO : get rid of auth headers from all requests
private const val VERSION = "0.5.0"
private const val CLIENT_TYPE = "CLIENT_APP"
private const val GRAPHQL_REQUEST_QUERY: String =
  "\nquery Session(" + "$" + "roomName: String!, " + "$" + "password: String) {\n\tsession(roomName: " + "$" + "roomName, password: " + "$" + "password) {\n\t  title,\n\t  roomNodeLink,\n\t  roomName,\n\t  password\n\t}\n  }\n"

internal class ApiClient(
  private val controllerContainer: IControllerContainer
) : IApiClient {

  internal val client: HttpClient = HttpClient {
    install(Logging) {
      logger = Logger.SIMPLE
      level =
        if (controllerContainer.platformUtilsProvider.getLogger().isLoggerEnabled()) ALL else NONE
    }
    install(ContentNegotiation) {
      json(Json {
        prettyPrint = true
        isLenient = true
        ignoreUnknownKeys = true
      })
    }
    install(HttpTimeout) {
      socketTimeoutMillis = 30000
      requestTimeoutMillis = 30000
      connectTimeoutMillis = 30000
    }
    defaultRequest {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
    }
  }

  val json = Json {
    ignoreUnknownKeys = true
  }

  override suspend fun getRoomNodeData(): MeetingSessionInfo {
    val roomName = controllerContainer.metaController.getRoomName()
    val graphQlRequestVariables = GraphQlRequestVariables(roomName, "")
    return if(controllerContainer.metaController.isV2AuthToken()) {
      val data = client.post("${controllerContainer.metaController.getBaseUrl()}/v2/internals/rooms").body<MeetingSessionDataWrapper>()
      MeetingSessionInfo(requireNotNull(data.data.title), requireNotNull(data.data.roomNodeLink))
    } else {
      val requestData = GraphQlRequest(GRAPHQL_REQUEST_QUERY, graphQlRequestVariables)
      val response = client.post("${controllerContainer.metaController.getBaseUrl()}/graphql") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        setBody(requestData)
        contentType(ContentType.Application.Json)
      }.body<MeetingSessionDataResponseV1>()
      MeetingSessionInfo(requireNotNull(response.data.session.title),
        requireNotNull(response.data.session.roomNodeLink))
    }
  }

  // TODO : fix this
  override suspend fun getICEServers(): IceServersWrapper {
    val response = client.get("${controllerContainer.metaController.getBaseUrl()}/iceservers") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
    }
    return response.body()
  }

  override suspend fun getParticipantInfo(): ParticipantInfo {
    return if (controllerContainer.metaController.isV2AuthToken()) {
      val userData =
        client.get("${controllerContainer.metaController.getBaseUrl()}/v2/internals/participant-details") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        }.body<UserDataWrapper>()
      ParticipantInfo.getParticipantInfoFromV2(userData)
    } else {
      val userDataWrapper =
        client.get("${controllerContainer.metaController.getBaseUrl()}/auth/basicUserDetails?authToken=${controllerContainer.metaController.getAuthToken()}")
          .body<UserDataWrapperV1>()
      val userPresetDataWrapper = getUserPreset()
      ParticipantInfo.getParticipantInfoFromV1(userDataWrapper, userPresetDataWrapper)
    }
  }

  override suspend fun getUserPreset(): UserPresetDataWrapper {
    val roomName = controllerContainer.metaController.getRoomName()
    val response = client.post("${controllerContainer.metaController.getBaseUrl()}/v1/userpreset") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      setBody(
        UserPresetRequestModel(
          controllerContainer.metaController.getAuthToken(),
          CLIENT_TYPE,
          roomName,
          VERSION
        )
      )
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun startRecording(): RecordingInfo {
    val recordingInfo: RecordingInfo
    if (controllerContainer.metaController.isV2AuthToken()) {
      val response =
        client.post("${controllerContainer.metaController.getBaseUrl()}/v2/recordings") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          setBody(
            StartRecordingModel(controllerContainer.metaController.getMeetingId())
          )
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapperV2>()
      recordingInfo = RecordingInfo(response.data.id)
    } else {
      val response =
        client.post("${controllerContainer.metaController.getBaseUrl()}/v1/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/recording") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapper>()
      recordingInfo = RecordingInfo(response.data.recording.id)
    }
    return recordingInfo
  }

  override suspend fun stopRecording(recordingInfo: RecordingInfo) {
    if (controllerContainer.metaController.isV2AuthToken()) {
      client.put("${controllerContainer.metaController.getBaseUrl()}/v2/recordings/${recordingInfo.id}") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        setBody(
          StopRecordingModelV2("stop")
        )
        contentType(ContentType.Application.Json)
      }
    } else {
      client.put("${controllerContainer.metaController.getBaseUrl()}/v1/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/recordings/${recordingInfo.id}") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        setBody(
          StopRecordingModel("stop")
        )
        contentType(ContentType.Application.Json)
      }
    }
  }

  override suspend fun getActiveRecording(): RecordingInfo {
    return if(controllerContainer.metaController.isV2AuthToken()) {
      val response =
        client.get("${controllerContainer.metaController.getBaseUrl()}/v2/recordings/active-recording/${controllerContainer.metaController.getMeetingId()}") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapperV2>()
      RecordingInfo(response.data.id)
    } else {
      val response =
        client.get("${controllerContainer.metaController.getBaseUrl()}/v1/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/active-recording/") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapper>()
      RecordingInfo(response.data.recording.id)
    }
  }

  override suspend fun getPlugins(): MultiplePluginResponse {
    val response = client.get("${controllerContainer.metaController.getBaseUrl()}/v2/plugins/user") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  // TODO : fix this endpoint
  override suspend fun getPluginDetails(pluginId: String): PluginResponse {
    val response =
      client.get("${controllerContainer.metaController.getBaseUrl()}/v2/plugins/view/$pluginId") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        contentType(ContentType.Application.Json)
      }
    return response.body()
  }

  // TODO : fix this endpoint
  override suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse {
    val response = client.get("${pluginBaseUrl}/dyte-config.json") {
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  // TODO : fix this endpoint
  override suspend fun authorizePlugin(pluginId: String): PluginAuthResponse {
    val response =
      client.post("${controllerContainer.metaController.getBaseUrl()}/v2/plugins/authorize/$pluginId") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        contentType(ContentType.Application.Json)
        setBody(
          PluginAuthorizationModel(
            controllerContainer.metaController.getRoomName(),
            controllerContainer.metaController.getPeerId()
          )
        )
      }
    return response.body()
  }

  override suspend fun uploadLogs(logs: PostDyteLogs) {
    client.post("https://api-silos.dyte.io/otel/logs") {
      contentType(ContentType.Application.Json)
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      setBody(logs)
    }
  }

  override fun getClient(): HttpClient {
    return client
  }
}

interface IApiClient {
  suspend fun getRoomNodeData(): MeetingSessionInfo
  suspend fun getParticipantInfo(): ParticipantInfo
  suspend fun getUserPreset(): UserPresetDataWrapper
  suspend fun getICEServers(): IceServersWrapper
  suspend fun startRecording(): RecordingInfo
  suspend fun stopRecording(recordingInfo: RecordingInfo)
  suspend fun getActiveRecording(): RecordingInfo
  suspend fun getPlugins(): MultiplePluginResponse
  suspend fun getPluginDetails(pluginId: String): PluginResponse
  suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse
  suspend fun authorizePlugin(pluginId: String): PluginAuthResponse
  suspend fun uploadLogs(logs: PostDyteLogs)
  fun getClient(): HttpClient
}