package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.platform.DytePluginFile
import io.dyte.core.platform.IDyteWebView
import io.dyte.core.platform.IDyteWebView.IDyteWebViewListener
import io.dyte.core.platform.WebView
import kotlinx.serialization.json.JsonObject

/**
 * Dyte Plugin
 *
 * @property controllerContainer
 * @constructor Create un-activated Dyte Plugin
 */
class DytePlugin(
  val id: String,
  val name: String,
  val description: String,
  val picture: String,
  val private: Boolean,
  val staggered: Boolean,
  val baseURL: String,
  private val controllerContainer: IControllerContainer
) {

  var enabledBy: String? = null

  var config: PluginConfigResponse? = null

  private var authToken: String? = null

  private val webViews: MutableMap<String, IDyteWebView> = mutableMapOf()

  private var _isActive: Boolean = false
  val isActive: Boolean
    get() = _isActive

  private val webViewListener: IDyteWebViewListener = object : IDyteWebViewListener {
    override fun onWebViewMessage(message: JsonObject) {
      controllerContainer.pluginsController.onPluginWebViewMessage(id, message)
    }

    override fun onFileRequest() {
      controllerContainer.pluginsController.onPluginFileRequest(this@DytePlugin)
    }
  }

  /**
   * Submits the event to the Plugin SDK through WebView.
   *
   * @param payload event JSON
   */
  internal fun submitEvent(payload: JsonObject) {
    webViews["default"]?.submitEvent(payload)
  }

  /**
   * Activate this plugin for all participants.
   */
  fun activate(): DyteActionResult {
    if (isActive.not()) {
      return controllerContainer.pluginsController.activatePlugin(this)
    }
    return DyteActionResult.Success
  }

  /**
   * Deactivate this plugin for all participants.
   */
  fun deactivate(): DyteActionResult {
    if (isActive) {
      return controllerContainer.pluginsController.deactivatePlugin(this)
    }
    return DyteActionResult.Success
  }

  @Throws(IllegalStateException::class)
  fun setPluginView(view: WebView) {
    if (authToken == null) throw IllegalStateException("No auth token set for plugin")
    webViews["default"]?.bindWebView(view)
  }

  fun removePluginView() {
    webViews["default"]?.unbindWebView()
  }

  /**
   * Uploads selected file to Plugin.
   */
  fun uploadFile(file: DytePluginFile) {
    webViews["default"]?.uploadFile(file)
  }

  /**
   * Enable this plugin for the local user.
   */
  internal fun enableLocal(
    authToken: String,
    webView: IDyteWebView,
    enabledBy: String
  ) {
    if (isActive) return

    this.authToken = authToken
    this.enabledBy = enabledBy
    webView.setListener(webViewListener)
    webViews["default"] = webView
    _isActive = true
  }

  /**
   * Disable this plugin for the local user.
   */
  internal fun disableLocal() {
    if (!isActive) return

    webViews["default"]?.removeListener()
    webViews.remove("default")
    _isActive = false
  }
}