package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.listeners.DyteParticipantUpdateListener

/**
 * Dyte meeting participant
 *
 * @property id The ID of the participant pertaining to local user
 * @property userId The userId of the participant.
 * @property name Contains Name of the local user.
 * @property picture The participant's picture (if any).
 * @property isHost
 * @property clientSpecificId  Identifier provided by the developer while adding the participant.
 * @property flags
 * @property audioEnabled A boolean value indicating if the audio currently enabled.
 * @property videoEnabled A boolean value indicating if the video currently enabled.
 * @constructor Create empty Dyte meeting participant
 */
abstract class DyteMeetingParticipant(
  open val id: String,
  open val userId: String,
  open val name: String,
  open val picture: String?,
  open val isHost: Boolean,
  open val clientSpecificId: String?,
  open val flags: ParticipantFlags,
  protected val controllerContainer: IControllerContainer
) {
  /**
   * The video track for the local user.
   */
  internal var _videoTrack: Any? = null
  val videoTrack: Any?
    get() = _videoTrack

  /**
   * The audio track for the local user.
   */
  // TODO : add audio track here
  // var audioTrack: Any? = null

  internal var _isScreenshareTrack = false
  val isScreenShareParticipant: Boolean
    get() = _isScreenshareTrack

  internal var _screenShareTrack: Any? = null
  val screenShareTrack: Any?
    get() = _screenShareTrack

  internal var _audioEnabled: Boolean = false
  val audioEnabled: Boolean
    get() = _audioEnabled

  internal var _videoEnabled: Boolean = false
  val videoEnabled: Boolean
    get() = _videoEnabled

  open fun disableVideo(): DyteActionResult {
    return controllerContainer.hostController.muteVideo(this)
  }

  open fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["id"] = id
    map["userId"] = userId
    map["name"] = name
    map["picture"] = picture
    map["isHost"] = isHost
    map["clientSpecificId"] = clientSpecificId
    map["flags"] = flags.toMap()
    map["audioEnabled"] = audioEnabled
    map["videoEnabled"] = videoEnabled
    return map
  }

  open fun disableAudio(): DyteActionResult {
    return controllerContainer.hostController.muteAudio(this)
  }

  open fun kick(): DyteActionResult {
    return controllerContainer.hostController.kick(id)
  }

  open fun acceptWaitListedRequest(): DyteActionResult {
    return controllerContainer.hostController.acceptWaitlistedRequest(this)
  }

  open fun rejectWaitListedRequest(): DyteActionResult {
    val result = controllerContainer.hostController.rejectWaitlistedRequest(this)
    if (result == DyteActionResult.Success) {
      controllerContainer.participantController.onWaitlistPeerRejected(this)
    }
    return result
  }

  open fun pin(): DyteActionResult {
    return controllerContainer.hostController.pinParticipant(this)
  }

  open fun unpin(): DyteActionResult {
    return controllerContainer.hostController.unpinParticipant()
  }

  fun addParticipantUpdateListener(participantUpdateListener: DyteParticipantUpdateListener) {
    controllerContainer.eventController.addParticipantUpdateListener(this, participantUpdateListener)
  }

  fun removeParticipantUpdateListener(participantUpdateListener: DyteParticipantUpdateListener) {
    controllerContainer.eventController.removeParticipantUpdateListener(this, participantUpdateListener)
  }

  fun getVideoView(): Any {
    return controllerContainer.participantController.getVideoView(this)
  }
}

data class ParticipantFlags(
  val recorder: Boolean,
  val hiddenParticipant: Boolean
) {
  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["recorder"] = recorder
    map["hiddenParticipant"] = hiddenParticipant
    return map
  }
}