package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnMeetingStateChanged
import io.dyte.core.socket.EVENT_CONNECT
import io.dyte.core.socket.EVENT_CONNECTING
import io.dyte.core.socket.EVENT_DISCONNECT
import io.dyte.core.socket.EVENT_ERROR
import io.dyte.core.socket.EVENT_MESSAGE
import io.dyte.core.socket.EVENT_PING
import io.dyte.core.socket.EVENT_PONG
import io.dyte.core.socket.EVENT_RECONNECT
import io.dyte.core.socket.EVENT_RECONNECT_ATTEMPT
import io.dyte.core.socket.Socket
import io.dyte.core.socket.SocketEvent.Connect
import io.dyte.core.socket.SocketEvent.Connecting
import io.dyte.core.socket.SocketEvent.Disconnect
import io.dyte.core.socket.SocketEvent.Error
import io.dyte.core.socket.SocketEvent.Message
import io.dyte.core.socket.SocketEvent.Ping
import io.dyte.core.socket.SocketEvent.Pong
import io.dyte.core.socket.SocketEvent.Reconnect
import io.dyte.core.socket.SocketEvent.ReconnectAttempt
import io.dyte.core.socket.SocketOptions
import io.dyte.core.socket.SocketOptions.Transport.WEBSOCKET
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ACTIVE_SPEAKER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_DISABLE_AUDIO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_DISABLE_VIDEO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_KICKED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_MUTE_ALL_AUDIO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_MUTE_ALL_VIDEO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_NO_ACTIVE_SPEAKER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_CHAT_MESSAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_CHAT_MESSAGES
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_POLL
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_POLLS
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_ADDED_TO_STAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_MUTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_PINNED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_REJECTED_TO_JOIN_STAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_UNMUTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_DATA
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_DISABLED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_ENABLED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_EVENT
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PRODUCER_CLOSED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_RECORDING_STARTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_RECORDING_STOPPED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_ACCEPTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_ADDED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_STOPPED_PRESENTING
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.BasePayloadModel
import io.dyte.core.socket.events.payloadmodel.OutboundMeetingEvents
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketActiveSpeakerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessage
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessagesModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConnectTransportModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarStagePeer
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginDisabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEnabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPoll
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollsModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerConnectModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAccepted
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerClosed
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerRejected
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarPresentRequestAccepted
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarRequestToJoinPeerAdded
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

private const val QUERY_PARAM_ROOM_URL = "roomURL"
private const val QUERY_PARAM_PEER_ID = "peerId"
private const val QUERY_PARAM_AUTH_TOKEN = "authToken"
private const val QUERY_PARAM_VERSION = "version"
private const val VERSION = "0.5.0"

internal class SocketController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), ISocketController {
  private lateinit var socket: Socket

  private var isSocketInReconnectState = false

  override fun init() {
    val meetingSessionData =
      runBlocking(Dispatchers.Default) { try {
        controllerContainer.apiClient.getRoomNodeData()
        } catch (e:Exception) {
          return@runBlocking null
        }
      }
    if (meetingSessionData != null) {
      controllerContainer.metaController.setMeetingTitle(meetingSessionData.title)
      val roomNodeLink = getRoomSocketLinkQueryParams(
        requireNotNull(meetingSessionData.roomNodeLink),
        controllerContainer.metaController.getPeerId(),
        controllerContainer.metaController.getMeetingId(),
        controllerContainer.metaController.getAuthToken()
      )

      val params = HashMap<String, String>().apply {
        put(QUERY_PARAM_ROOM_URL, controllerContainer.metaController.getMeetingId())
        put(QUERY_PARAM_PEER_ID, controllerContainer.metaController.getPeerId())
        put(QUERY_PARAM_AUTH_TOKEN, controllerContainer.metaController.getAuthToken())
        put(QUERY_PARAM_VERSION, VERSION)
        put("EIO", "4")
      }

      socket = Socket(
        endpoint = roomNodeLink,
        config = SocketOptions(
          transport = WEBSOCKET,
          queryParams = params
        )
      ) {
        on(Connect) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_CONNECT,
              isSocketInReconnectState
            )
          )
          if (isSocketInReconnectState) {
            isSocketInReconnectState = false
          } else {
            controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomInitCompleted)
          }
        }

        on(Connecting) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_CONNECTING
            )
          )
        }

        on(Disconnect) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_DISCONNECT
            )
          )
        }

        on(Error) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_ERROR
            )
          )
        }

        on(Reconnect) {
          isSocketInReconnectState = true
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_RECONNECT
            )
          )
        }

        on(ReconnectAttempt) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_RECONNECT_ATTEMPT
            )
          )
        }

        on(Ping) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_PING
            )
          )
        }

        on(Pong) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_PONG
            )
          )
        }

        on(Message) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_MESSAGE
            )
          )
        }

        on("event://server-simple-message") { response, onDone ->
          try {
            val parsedResponse =
              controllerContainer.socketMessageResponseParser.parseResponse(response)
            controllerContainer.loggerController.traceLog(parsedResponse.eventType.name)
            handleEvent(parsedResponse.eventType, parsedResponse.payload, onDone)
          } catch (ignored: Exception) {
            // ignore
          }
        }
      }
    } else {
      controllerContainer.loggerController.traceError("Socket init failed")
      throw IllegalStateException("Unable to init socket")
    }
  }

  override fun connect() {
    socket.connect()
  }

  override fun sendMessageSync(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?
  ): String {
    val eventData = createEventData(outboundMeetingEventType, payload).toString()
    val response = runBlocking(Dispatchers.Default) {
      socket.emitAck(
        OutboundMeetingEvents.SEND_MESSAGE.eventName,
        eventData
      )
    }
    return requireNotNull(response)
  }

  override fun sendMessageSyncGeneric(
    payload: JsonObject
  ): String {
    val response = runBlocking(Dispatchers.Default) {
      socket.emitAck(
        OutboundMeetingEvents.SEND_MESSAGE.eventName,
        payload.toString()
      )
    }
    return requireNotNull(response)
  }

  override fun disconnect() {
    socket.disconnect()
  }

  private fun handleEvent(
    inboundMeetingEventType: InboundMeetingEventType,
    payloadModel: BasePayloadModel,
    onDone: () -> Unit
  ) {
    when (inboundMeetingEventType) {
      InboundMeetingEventType.WEB_SOCKET_PEER_JOINED -> handlePeerJoined(payloadModel as WebSocketMeetingPeerUser)
      InboundMeetingEventType.WEB_SOCKET_PEER_LEFT -> handlePeerLeft(payloadModel as WebSocketPeerLeftModel)

      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_ADDED -> handleWaitlistPeerAdded(payloadModel as WebSocketWaitlistPeerAdded)
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_ACCEPTED -> handleWaitlistPeerAccepted(
        payloadModel as WebSocketWaitlistPeerAccepted
      )
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_REJECTED -> handleWaitlistPeerRejected(
        payloadModel as WebSocketWaitlistPeerRejected
      )
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_CLOSED -> handleWaitlistPeerClosed(
        payloadModel as WebSocketWaitlistPeerClosed
      )

      InboundMeetingEventType.WEB_SOCKET_SELECTED_PEERS -> handleSelectedPeers(payloadModel as WebSocketSelectedPeersModel)
      InboundMeetingEventType.WEB_SOCKET_NEW_CONSUMER -> handleNewConsumer(
        payloadModel as WebSocketConsumerModel,
        onDone
      )
      InboundMeetingEventType.WEB_SOCKET_RESUME_CONSUMER -> handleResumeConsumer(payloadModel as WebSocketConsumerResumedModel)
      InboundMeetingEventType.WEB_SOCKET_CLOSE_CONSUMER -> handleCloseConsumer(payloadModel as WebSocketConsumerClosedModel)
      InboundMeetingEventType.WEB_SOCKET_PAUSE_CONSUMER -> handlePauseConsumer(payloadModel as WebSocketConsumerClosedModel)
      InboundMeetingEventType.WEB_SOCKET_PRODUCER_CONNECT -> handleProducerConnect(payloadModel as WebSocketProducerConnectModel)
      InboundMeetingEventType.WEB_SOCKET_CONNECT_TRANSPORT -> handleConnectTransport(
        payloadModel as WebSocketConnectTransportModel,
        onDone
      )
      WEB_SOCKET_PEER_MUTED -> handlePeerMuted(payloadModel as WebSocketPeerMuteModel)
      WEB_SOCKET_PEER_UNMUTED -> handlePeerUnMuted(payloadModel as WebSocketPeerMuteModel)
      WEB_SOCKET_PRODUCER_CLOSED -> handleProducerClosed(payloadModel as WebSocketProducerClosedModel)
      WEB_SOCKET_ON_CHAT_MESSAGE -> handleChatMessage(payloadModel as WebSocketChatMessage)
      WEB_SOCKET_ON_CHAT_MESSAGES -> handleChatMessages(payloadModel as WebSocketChatMessagesModel)
      WEB_SOCKET_ON_POLLS -> handlePolls(payloadModel as WebSocketPollsModel)
      WEB_SOCKET_ON_POLL -> handlePoll(payloadModel as WebSocketPollModel)
      WEB_SOCKET_MUTE_ALL_VIDEO -> handleMuteAllVideo()
      WEB_SOCKET_MUTE_ALL_AUDIO -> handleMuteAllAudio()
      WEB_SOCKET_ACTIVE_SPEAKER -> {
        handleActiveSpeaker(payloadModel as WebSocketActiveSpeakerModel)
      }
      WEB_SOCKET_NO_ACTIVE_SPEAKER -> {
        handleNoActiveSpeaker()
      }
      WEB_SOCKET_PEER_PINNED -> {
        handlePeerPinned(payloadModel as WebSocketPeerPinnedModel)
      }
      WEB_SOCKET_RECORDING_STARTED -> {
        handleRecordingStarted()
      }
      WEB_SOCKET_RECORDING_STOPPED -> {
        handleRecordingStopped()
      }
      WEB_SOCKET_KICKED -> {
        handleKicked()
      }
      WEB_SOCKET_DISABLE_AUDIO -> {
        handleDisableAudio()
      }
      WEB_SOCKET_DISABLE_VIDEO -> {
        handleDisableVideo()
      }
      WEB_SOCKET_PLUGIN_ENABLED -> {
        handlePluginEnabled(payloadModel as WebSocketPluginEnabled)
      }
      WEB_SOCKET_PLUGIN_DISABLED -> {
        handlePluginDisabled(payloadModel as WebSocketPluginDisabled)
      }
      WEB_SOCKET_PLUGIN_DATA, WEB_SOCKET_PLUGIN_EVENT -> {
        handlePluginEvent(inboundMeetingEventType.type, payloadModel as WebSocketPluginEvent)
      }
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_ACCEPTED -> {
        handlePresentOnStageRequestReceived(payloadModel as WebSocketWebinarPresentRequestAccepted)
      }
      WEB_SOCKET_PEER_ADDED_TO_STAGE -> {
        handlePeerAddedToStage(payloadModel as WebSocketWebinarStagePeer)
      }
      WEB_SOCKET_PEER_REJECTED_TO_JOIN_STAGE -> {
        handlePeerRejectedToStage(payloadModel as WebSocketWebinarStagePeer)
      }
      WEB_SOCKET_STOPPED_PRESENTING -> {
        handleStoppedPresenting()
      }
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_ADDED -> {
        requestToJoinPeerAdded(payloadModel as WebSocketWebinarRequestToJoinPeerAdded)
      }
      else -> {
        // no-op
      }
    }
  }

  private fun requestToJoinPeerAdded(webSocketRequestPeerAddedToStage: WebSocketWebinarRequestToJoinPeerAdded) {
    controllerContainer.webinarController.onRequestToPresentPeerAdded(RequestToPresentParticipant(webSocketRequestPeerAddedToStage.id, webSocketRequestPeerAddedToStage.name, webSocketRequestPeerAddedToStage.requestToJoinType))
  }

  private fun handleStoppedPresenting() {
    controllerContainer.webinarController.onStoppedPresenting()
  }

  private fun handlePeerAddedToStage(webSocketWebinarStagePeer: WebSocketWebinarStagePeer) {
    controllerContainer.webinarController.onPeerAddedToStage(webSocketWebinarStagePeer.id)
  }

  private fun handlePeerRejectedToStage(webSocketWebinarStagePeer: WebSocketWebinarStagePeer) {
    controllerContainer.webinarController.onPeerRejectedToStage(webSocketWebinarStagePeer.id)
  }

  private fun handlePresentOnStageRequestReceived(payload: WebSocketWebinarPresentRequestAccepted) {
    controllerContainer.webinarController.onRequestedToPresent(payload.requestToJoinType)
  }

  private fun handleDisableVideo() {
    controllerContainer.selfController.disableVideo()
  }

  private fun handleDisableAudio() {
    controllerContainer.selfController.disableAudio()
  }

  private fun handleKicked() {
    controllerContainer.platformUtilsProvider.getMediaSoupUtils().leaveCall()
    disconnect()
    controllerContainer.eventController.triggerEvent(DyteEventType.OnHostKicked)
  }

  private fun handleRecordingStopped() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStopped)
  }

  private fun handleRecordingStarted() {
    controllerContainer.recordingController.fetchRecordingData()
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
  }

  private fun handlePeerPinned(websocketPeerPinnedModel: WebSocketPeerPinnedModel) {
    if (websocketPeerPinnedModel.peerId != null) {
      controllerContainer.participantController.onPeerPinned(websocketPeerPinnedModel)
    } else {
      controllerContainer.participantController.onPeerUnpinned()
    }
  }

  private fun handleNoActiveSpeaker() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnNoActiveSpeaker)
  }

  private fun handleActiveSpeaker(webSocketActiveSpeakerModel: WebSocketActiveSpeakerModel) {
    val participant =
      controllerContainer.participantController.meetingRoomParticipants.joined.find { it.id == webSocketActiveSpeakerModel.peerId }
    participant?.let {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnActiveSpeakerChanged(
          participant
        )
      )
    }
  }

  private fun createEventData(type: OutboundMeetingEventType?, payload: JsonElement?): JsonObject {
    val mapper = hashMapOf<String, JsonElement>()
    type?.type?.let { mapper.put("type", JsonPrimitive(it)) }
    payload?.let { mapper.put("payload", payload) }
    return JsonObject(mapper)
  }

  /**
   * Get room socket link query params
   *
   * adds roomName, authToken, peerId to the room node link, we receive in the meeting session graphql api
   *
   * @param roomNodeLink
   * @param peerId
   * @param meetingName
   * @param authToken
   * @return
   */
  private fun getRoomSocketLinkQueryParams(
    roomNodeLink: String,
    peerId: String,
    meetingName: String,
    authToken: String
  ): String {
    val params = HashMap<String, String>().apply {
      put(QUERY_PARAM_ROOM_URL, controllerContainer.platformUtilsProvider.getPlatformUtils().getUrlEncodedString(meetingName))
      put(QUERY_PARAM_PEER_ID, peerId)
      put(QUERY_PARAM_AUTH_TOKEN, authToken)
      put(QUERY_PARAM_VERSION, VERSION)
      put("EIO", "4")
    }

    val map = HashMap<String, String>()
    params.forEach {
      map[it.key] = it.value
    }

    val urlBuilder = StringBuilder(roomNodeLink)

    if (!map.isEmpty()) {
      urlBuilder.append("?")

      map.keys.forEachIndexed { index, key ->
        urlBuilder.append(key).append("=").append(map[key])

        if (index < map.keys.size - 1) {
          urlBuilder.append("&")
        }
      }
    }

    return urlBuilder.toString()
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed) {
    controllerContainer.participantController.onWaitlistPeerClosed(webSocketWaitlistPeerClosed)
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected) {
    controllerContainer.participantController.onWaitlistPeerRejected(
      webSocketWaitlistPeerRejected
    )
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted) {
    controllerContainer.participantController.onWaitlistPeerAccepted(
      webSocketWaitlistPeerAccepted
    )
  }

  private fun handleWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded) {
    controllerContainer.participantController.onWaitlistPeerAdded(webSocketWaitlistPeerAdded)
  }

  private fun handlePoll(webSocketPollModel: WebSocketPollModel) {
    handleSocketPoll(webSocketPollModel.poll)
  }

  private fun handleMuteAllVideo() {
    controllerContainer.selfController.disableVideo()
  }

  private fun handleMuteAllAudio() {
    controllerContainer.selfController.disableAudio()
  }

  private fun handleSocketPoll(webSocketPoll: WebSocketPoll) {
    controllerContainer.pollsController.onNewPoll(webSocketPoll)
  }

  private fun handleChatMessages(webSocketChatMessagesModel: WebSocketChatMessagesModel) {
    controllerContainer.chatController.handleChatMessages(webSocketChatMessagesModel)
  }

  private fun handleChatMessage(webSocketChatMessage: WebSocketChatMessage) {
    controllerContainer.chatController.handleNewChatMessage(webSocketChatMessage)
  }

  private fun handlePolls(webSocketPollsModel: WebSocketPollsModel) {
    webSocketPollsModel.polls?.values?.forEach {
      handleSocketPoll(it)
    }
  }

  private fun handlePeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel) {
    controllerContainer.participantController.onPeerLeft(webSocketPeerLeftModel)
  }

  private fun handleCloseConsumer(payloadModel: WebSocketConsumerClosedModel) {
    controllerContainer.participantController.onParticipantVideoMuted(payloadModel)
    controllerContainer.platformUtilsProvider.getMediaSoupUtils().handleCloseConsumer(payloadModel)
  }

  private fun handlePauseConsumer(payloadModel: WebSocketConsumerClosedModel) {
    controllerContainer.participantController.onParticipantVideoMuted(payloadModel)
  }

  private fun handleResumeConsumer(payloadModel: WebSocketConsumerResumedModel) {
    controllerContainer.participantController.onParticipantVideoUnmuted(payloadModel)
  }

  private fun handlePeerJoined(meetingPeerUser: WebSocketMeetingPeerUser) {
    controllerContainer.participantController.onPeerJoined(meetingPeerUser)
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel) {
    controllerContainer.participantController.onSelectedPeers(webSocketSelectedPeersModel)
  }

  private fun handleConnectTransport(
    payloadModel: WebSocketConnectTransportModel,
    onDone: () -> Unit
  ) {
    println("DyteMobileClient | SocketController handleConnectTransport ")
    controllerContainer.platformUtilsProvider.getMediaSoupUtils().connectTransport(
      payloadModel.id ?: return,
      payloadModel.producing ?: false,
      onDone
    )
  }

  private fun handleNewConsumer(
    webSocketConsumerModel: WebSocketConsumerModel,
    onDone: () -> Unit
  ) {
    controllerContainer.mediaSoupController.onNewConsumer(webSocketConsumerModel)
    controllerContainer.platformUtilsProvider.getMediaSoupUtils()
      .handleNewConsumer(webSocketConsumerModel, onDone)
  }

  private fun handlePeerMuted(payloadModel: WebSocketPeerMuteModel) {
    if (payloadModel.peerId == controllerContainer.metaController.getPeerId()) {
      controllerContainer.selfController.onAudioDisabled()
    } else {
      controllerContainer.participantController.onPeerAudioMuted(payloadModel)
    }
  }

  private fun handlePeerUnMuted(payloadModel: WebSocketPeerMuteModel) {
    if (payloadModel.peerId == controllerContainer.metaController.getPeerId()) {
      controllerContainer.selfController.onAudioEnabled()
    } else {
      controllerContainer.participantController.onPeerAudioUnmuted(payloadModel)
    }
  }

  private fun handleProducerConnect(payloadModel: WebSocketProducerConnectModel) {
    controllerContainer.selfController.onVideoEnabled(requireNotNull(payloadModel.id))
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleProducerClosed(payloadModel: WebSocketProducerClosedModel) {
    controllerContainer.selfController.onVideoDisabled()
  }

  /* Plugin message handlers */
  private fun handlePluginEnabled(payloadModel: WebSocketPluginEnabled) {
    controllerContainer.pluginsController.onEnablePlugin(payloadModel)
  }

  private fun handlePluginDisabled(payloadModel: WebSocketPluginDisabled) {
    controllerContainer.pluginsController.onDisablePlugin(payloadModel)
  }

  private fun handlePluginEvent(eventType: String, payloadModel: WebSocketPluginEvent) {
    controllerContainer.pluginsController.onPluginSocketEvent(eventType, payloadModel)
  }
}

interface ISocketController {
  fun connect()
  fun disconnect()

  fun sendMessageSync(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?
  ): String

  fun sendMessageSyncGeneric(
    payload: JsonObject
  ): String
}
