package io.dyte.core.controllers

import io.dyte.core.controllers.DyteRecordingState.IDLE
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.controllers.DyteRecordingState.STARTING
import io.dyte.core.controllers.DyteRecordingState.STOPPING
import io.dyte.core.models.DyteActionResult
import io.dyte.core.network.info.RecordingInfo
import io.dyte.core.network.models.RecordingData
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlin.Exception

internal class RecordingController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), IRecordingController {
  private lateinit var recordingInfo: RecordingInfo
  private var recordingState: DyteRecordingState = IDLE

  override fun init() {
    println("DyteMobileClient | RecordingController init ")
    try {
      recordingInfo = runBlocking(Dispatchers.Default) {
        controllerContainer.apiClient.getActiveRecording()
      }
      recordingState = RECORDING
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
    } catch (e:Exception) {
      println("DyteMobileClient | RecordingController init ${e.message}")
      e.printStackTrace()
    }
  }

  override suspend fun start() : DyteActionResult{
    if (controllerContainer.presetController.canRecord().not()) {
      return DyteActionResult.ActionNotPermitted
    }

    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        try {
          recordingState = STARTING
          recordingInfo = controllerContainer.apiClient.startRecording()
          recordingState = RECORDING
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
        } catch (e:Exception) {
          e.printStackTrace()
          controllerContainer.loggerController.traceError("error in starting recording ${e.message}")
          recordingState = IDLE
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
        }
      }
    }
    return DyteActionResult.Success
  }

  override suspend fun stop() : DyteActionResult{
    if (controllerContainer.presetController.canRecord().not()) {
      return DyteActionResult.ActionNotPermitted
    }
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        try {
          recordingState = STOPPING
          controllerContainer.apiClient.stopRecording(recordingInfo)
          recordingState = IDLE
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStopped)
        } catch (e:Exception) {
          e.printStackTrace()
          controllerContainer.loggerController.traceError("error in stopping recording ${e.message}")
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
        }
      }
    }
    return DyteActionResult.Success
  }

  override fun getState(): DyteRecordingState {
    return recordingState
  }

  override fun fetchRecordingData() {
    recordingInfo = runBlocking(Dispatchers.Default) {
      controllerContainer.apiClient.getActiveRecording()
    }
  }
}

interface IRecordingController {
  suspend fun start(): DyteActionResult
  suspend fun stop(): DyteActionResult
  fun getState(): DyteRecordingState

  fun fetchRecordingData()
}

enum class DyteRecordingState {
  IDLE,
  STARTING,
  RECORDING,
  STOPPING;
}