package io.dyte.core.controllers

import io.dyte.core.models.DyteActionResult
import io.dyte.core.models.DytePlugin
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginDisabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEnabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEvent
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.ktor.http.URLBuilder
import io.ktor.http.set
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.*

internal class PluginsController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), IPluginsController {

  private val _allPlugins = mutableMapOf<String, DytePlugin>()
  private val _activePlugins = mutableMapOf<String, DytePlugin>()

  override val allPlugins: List<DytePlugin>
    get() = _allPlugins.values.toList()

  override val activePlugins: List<DytePlugin>
    get() = _activePlugins.values.toList()

  override fun onEnablePlugin(payload: WebSocketPluginEnabled) {
    if (payload.id == null || payload.enabledBy == null) return

    // Return if plugin is already active
    _activePlugins[payload.id]?.let { plugin ->
      if (plugin.isActive) return
    }

    enablePluginForLocalUser(payload.id!!, payload.enabledBy!!)

    _activePlugins[payload.id]?.let { plugin ->
      if (plugin.isActive) {
        controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginEnabled(plugin))
      }
    }
  }

  override fun onDisablePlugin(payload: WebSocketPluginDisabled) {
    val plugin = _activePlugins[payload.id] ?: return
    plugin.disableLocal()
    _activePlugins.remove(plugin.id)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginDisabled(plugin))
  }

  override fun onPluginSocketEvent(type: String, payload: WebSocketPluginEvent) {
    val pluginId = payload.payload["id"]?.jsonPrimitive?.content ?: return
    val pluginPayload = buildJsonObject {
      put("type", type)
      put("payload", payload.payload)
    }
    submitEventToPlugin(pluginId, pluginPayload)
  }

  override fun onPluginWebViewMessage(pluginId: String, message: JsonObject) {
    val type = message["type"]?.jsonPrimitive?.content
    when (type) {
      "pluginEvent",
      "storePluginData",
      "getPluginData",
      "enablePluginForAll",
      "enablePluginForUser",
      "disablePluginForUser",
      "chatMessage" -> {
        val response = controllerContainer.socketController.sendMessageSyncGeneric(message)
        val jsonResponse = Json { isLenient = true }.parseToJsonElement(response)
        submitEventToPlugin(pluginId, jsonResponse.jsonObject)
      }
      "getJoinedPeers" -> {
        val roomStateResponse =
          controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
        val roomState = (controllerContainer.socketMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState ?: return
        val peers = roomState.peers ?: emptyList()
        val pluginEvent = buildJsonObject {
          put("type", "websocket/get-joined-peers")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("peers", Json { encodeDefaults = true }.encodeToJsonElement(peers))
          }
        }

        submitEventToPlugin(pluginId, pluginEvent)
      }
      "getPeerInfo" -> {
        val peerId = message["payload"]?.jsonObject?.get("peerId")?.jsonPrimitive?.content
          ?: controllerContainer.selfController.getSelf().id
        val socketPayload = buildJsonObject {
          put("type", type)
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("peerId", peerId)
          }
        }
        val response = controllerContainer.socketController.sendMessageSyncGeneric(socketPayload)
        val jsonResponse = Json { isLenient = true }.parseToJsonElement(response)
        submitEventToPlugin(pluginId, jsonResponse.jsonObject)
      }
      "getRoomState" -> {
        val roomStateResponse =
          controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
        val roomState = (controllerContainer.socketMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState ?: return

        val peers = roomState.peers ?: emptyList()
        val enabledBy = roomState.plugins?.find { it.id == pluginId }?.enabledBy ?: return
        val enablerPeerId = peers.find { it.userId == enabledBy }?.id ?: return

        val payload = buildJsonObject {
          put("type", "websocket/get-room-state")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("roomName", roomState.roomName)
            put("pluginInitiatorPeerId", enablerPeerId)
            put("pluginInitiatorUserId", enabledBy)
            put("peers", Json { encodeDefaults = true }.encodeToJsonElement(peers))
            put("displayTitle", roomState.displayTitle)
            put("roomUUID", roomState.roomUUID)
            put("currentPeer", controllerContainer.selfController.getSelf().id)
          }
        }
        submitEventToPlugin(pluginId, payload)
      }
      "getPluginInitiatorId" -> {
        val roomStateResponse =
          controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
        val roomState = (controllerContainer.socketMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState ?: return

        val peers = roomState.peers ?: emptyList()
        val enabledBy = roomState.plugins?.find { it.id == pluginId }?.enabledBy ?: return
        val enablerPeerId = peers.find { it.userId == enabledBy }?.id ?: return

        val payload = buildJsonObject {
          put("type", "websocket/get-plugin-initiator-peer-id")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("pluginInitiatorPeerId", enablerPeerId)
          }
        }
        submitEventToPlugin(pluginId, payload)
      }
      "getPluginInitiatorUserId" -> {
        val roomStateResponse =
          controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
        val roomState = (controllerContainer.socketMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState ?: return

        val enabledBy = roomState.plugins?.find { it.id == pluginId }?.enabledBy ?: return

        val payload = buildJsonObject {
          put("type", "websocket/get-plugin-initiator-user-id")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("pluginInitiatorUserId", enabledBy)
          }
        }
        submitEventToPlugin(pluginId, payload)
      }
      "pluginEventToParent" -> {}
      else -> {
        // no-op
      }
    }
  }

  override fun activatePlugin(plugin: DytePlugin) : DyteActionResult{
    if (controllerContainer.presetController.permissions.pluginPermissions.canStart.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    runBlocking(Dispatchers.Default) {
      val activatePluginPayload = buildJsonObject {
        put("id", plugin.id)
        put("staggered", plugin.staggered)
      }
      controllerContainer.socketController.sendMessageSync(
        OutboundMeetingEventType.ADD_ROOM_PLUGIN,
        activatePluginPayload
      )
    }
    return DyteActionResult.Success
  }

  override fun deactivatePlugin(plugin: DytePlugin) : DyteActionResult{
    if (controllerContainer.presetController.permissions.pluginPermissions.canClose.not()) {
      return DyteActionResult.ActionNotPermitted
    }
    runBlocking(Dispatchers.Default) {
      val deactivatePluginPayload = buildJsonObject {
        put("id", plugin.id)
      }
      controllerContainer.socketController.sendMessageSync(
        OutboundMeetingEventType.REMOVE_ROOM_PLUGIN,
        deactivatePluginPayload
      )
    }
    return DyteActionResult.Success
  }

  override fun handleRoomState(roomState: WebSocketRoomStateModel) {
    roomState.roomState?.plugins?.forEach { roomPlugin ->
      val plugin = _allPlugins[roomPlugin.id]
      plugin?.let {
        enablePluginForLocalUser(plugin.id, roomPlugin.enabledBy)
      }
    }
  }

  override fun onPluginFileRequest(plugin: DytePlugin) {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginFileRequest(plugin))
  }

  private fun enablePluginForLocalUser(pluginId: String, enabledBy: String) {
    // for safe-guard
    if (_activePlugins.containsKey(pluginId)) return

    val plugin = _allPlugins[pluginId] ?: return

    // just for safe-guard
    if (plugin.isActive) return

    runBlocking(Dispatchers.Default) {
      try {
        val pluginAuthToken = controllerContainer.apiClient.authorizePlugin(plugin.id).data.token
        val pluginWebView = controllerContainer.platformUtilsProvider.createWebView()
        val pluginUrl = URLBuilder(plugin.baseURL)
        pluginUrl.set {
          parameters["auth"] = pluginAuthToken
          parameters["pluginId"] = plugin.id
          parameters["backend"] = controllerContainer.metaController.getBaseUrl()
          encodedParameters["parent"] = "*"
        }
        pluginWebView.setURL(pluginUrl.buildString())
        plugin.enableLocal(
          pluginAuthToken,
          pluginWebView,
          enabledBy
        )
        _activePlugins[plugin.id] = plugin
      } catch (e: Exception) {
        // no-op
      }
    }
  }

  override fun init() {
    runBlocking(Dispatchers.Default) {
      val response = controllerContainer.apiClient.getPlugins()
      response.data.plugins.forEach {
        val plugin = DytePlugin(
          id = it.id,
          name = it.name,
          picture = it.picture,
          description = it.description,
          private = it.private,
          staggered = it.staggered,
          baseURL = it.baseURL,
          controllerContainer = controllerContainer
        )
        _allPlugins[plugin.id] = plugin
      }
    }
  }

  private fun submitEventToPlugin(pluginId: String, payload: JsonObject) {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      _activePlugins[pluginId]?.submitEvent(payload) ?: return@runOnMainThread
    }
  }
}

interface IPluginsController {
  val allPlugins: List<DytePlugin>

  val activePlugins: List<DytePlugin>

  fun onEnablePlugin(payload: WebSocketPluginEnabled)

  fun onDisablePlugin(payload: WebSocketPluginDisabled)

  fun onPluginSocketEvent(type: String, payload: WebSocketPluginEvent)

  fun onPluginWebViewMessage(pluginId: String, message: JsonObject)

  fun activatePlugin(plugin: DytePlugin): DyteActionResult

  fun deactivatePlugin(plugin: DytePlugin): DyteActionResult

  fun handleRoomState(roomState: WebSocketRoomStateModel)

  fun onPluginFileRequest(plugin: DytePlugin)
}
